package pangea.hiagent.tool;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import pangea.hiagent.workpanel.event.SseEventBroadcaster;
import pangea.hiagent.web.dto.WorkPanelEvent;

import java.util.HashMap;
import java.util.Map;

/**
 * 所有工具类的基础抽象类
 * 提供工具执行监控和SSE事件发送功能
 */
@Slf4j
public abstract class BaseTool {
    
    @Autowired
    private SseEventBroadcaster sseEventBroadcaster;
    
    /**
     * 工具执行包装方法
     * 监控工具方法的完整执行生命周期
     * @param methodName 被调用的方法名称
     * @param params 方法参数映射
     * @param action 实际执行的工具逻辑
     * @param <T> 返回类型
     * @return 工具执行结果
     */
    protected <T> T execute(String methodName, Map<String, Object> params, ToolAction<T> action) {
        String toolName = this.getClass().getSimpleName();
        long startTime = System.currentTimeMillis();
        
        // 1. 发送工具开始执行事件
        sendToolEvent(toolName, methodName, params, null, "执行中", startTime, null);
        
        T result = null;
        String status = "成功";
        Exception exception = null;
        
        try {
            // 2. 执行实际的工具逻辑
            result = action.run();
        } catch (Exception e) {
            status = "失败";
            exception = e;
            throw new RuntimeException("工具执行失败: " + e.getMessage(), e);
        } finally {
            // 记录结束时间和耗时
            long endTime = System.currentTimeMillis();
            long duration = endTime - startTime;
            
            // 3. 发送工具执行完成事件
            sendToolEvent(toolName, methodName, params, result, status, startTime, duration, exception);
        }
        
        return result;
    }
    
    /**
     * 简化版execute方法，无需手动构建参数映射
     * @param methodName 被调用的方法名称
     * @param action 实际执行的工具逻辑
     * @param <T> 返回类型
     * @return 工具执行结果
     */
    protected <T> T execute(String methodName, ToolAction<T> action) {
        return execute(methodName, new HashMap<>(), action);
    }
    
    /**
     * 发送工具事件给前端
     * @param toolName 工具名称
     * @param methodName 方法名称
     * @param params 参数信息
     * @param result 执行结果
     * @param status 执行状态（执行中/成功/失败）
     * @param startTime 开始时间戳
     * @param duration 执行耗时（毫秒）
     * @param exception 异常信息（可选）
     */
    private void sendToolEvent(String toolName, String methodName, 
                              Map<String, Object> params, Object result, String status, 
                              Long startTime, Long duration, Exception... exception) {
        try {
            Map<String, Object> eventData = new HashMap<>();
            eventData.put("toolName", toolName);
            eventData.put("methodName", methodName);
            eventData.put("params", params);
            eventData.put("result", result);
            eventData.put("status", status);
            eventData.put("startTime", startTime);
            eventData.put("duration", duration);
            
            if (exception != null && exception.length > 0 && exception[0] != null) {
                eventData.put("error", exception[0].getMessage());
                eventData.put("errorType", exception[0].getClass().getSimpleName());
            }
            
            WorkPanelEvent event = WorkPanelEvent.builder()
                    .type("tool_call")
                    .title(toolName + "." + methodName)
                    .timestamp(System.currentTimeMillis())
                    .metadata(eventData)
                    .build();
            
            sseEventBroadcaster.broadcastWorkPanelEvent(event);
            
            log.debug("已发送工具事件: {}#{}, 状态: {}", toolName, methodName, status);
        } catch (Exception e) {
            log.error("发送工具事件失败: {}", e.getMessage(), e);
        }
    }
    
    /**
     * 工具动作函数式接口
     * 用于封装实际要执行的工具逻辑
     * @param <T> 返回类型
     */
    @FunctionalInterface
    protected interface ToolAction<T> {
        T run() throws Exception;
    }
}