package pangea.hiagent.tool.impl;

import com.microsoft.playwright.*;
import com.microsoft.playwright.options.LoadState;
import com.microsoft.playwright.options.WaitUntilState;
import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.tool.annotation.Tool;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.io.File;
import java.nio.file.Paths;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.HashMap;
import java.util.Map;

import pangea.hiagent.tool.BaseTool;
import pangea.hiagent.web.service.ToolConfigService;
import pangea.hiagent.workpanel.playwright.PlaywrightManager;

/**
 * 海信LBPM流程审批工具类
 * 专门负责LBPM流程审批功能，需要先通过HisenseSsoLoginTool登录
 * 该工具专注于流程审批操作，不处理登录逻辑
 */
@Slf4j
@Component
public class HisenseLbpmApprovalTool extends BaseTool {

    // SSO登录页面URL
    private static final String SSO_LOGIN_URL = "https://sso.hisense.com/login/";

    // 注入Playwright管理器
    @Autowired
    private PlaywrightManager playwrightManager;
    

    @Autowired
    private ToolConfigService toolConfigService;
    
    // 存储目录路径
    private static final String STORAGE_DIR = "storage";

    
    /**
     * 从数据库获取SSO用户名
     * 
     * @return SSO用户名
     */
    public String getSsoUsername() {
        log.debug("从数据库获取SSO用户名");
        return toolConfigService.getParamValue("hisenseSsoLogin", "ssoUsername");
    }

    /**
     * 工具方法：处理海信请假审批
     * 
     * @param userId 用户ID，用于区分不同用户的会话
     * @param approvalUrl 请假审批页面URL
     * @param approvalOpinion 审批意见
     * @return 处理结果
     */
    @Tool(description = "处理海信请假审批、自驾车审批、调休审批，需要先使用HisenseSsoLoginTool登录，提供用户ID以区分会话")
    public String processHisenseLeaveApproval(String approvalUrl, String approvalOpinion) {
        Map<String, Object> params = new HashMap<>();
        params.put("approvalUrl", approvalUrl);
        params.put("approvalOpinion", approvalOpinion);
        
        return execute("processHisenseLeaveApproval", params, () -> {
            String ssoUsername = getSsoUsername();
            log.info("开始为用户 {} 处理海信请假审批，URL: {}", ssoUsername, approvalUrl);
            
            // 参数校验
            if (ssoUsername == null || ssoUsername.isEmpty()) {
                String errorMsg = "用户ID不能为空";
                log.error(errorMsg);
                return errorMsg;
            }
            
            if (approvalUrl == null || approvalUrl.isEmpty()) {
                String errorMsg = "审批URL不能为空";
                log.error(errorMsg);
                return errorMsg;
            }
            
            if (approvalOpinion == null || approvalOpinion.isEmpty()) {
                String errorMsg = "审批意见不能为空";
                log.error(errorMsg);
                return errorMsg;
            }
            
            Page page = null;
            
            try {
                    // 获取用户专用的浏览器上下文
                BrowserContext userContext = playwrightManager.getUserContext(ssoUsername);
                
                // 创建新页面
                page = userContext.newPage();
                
                // 访问审批页面
                log.info("正在访问审批页面: {}", approvalUrl);
                page.navigate(approvalUrl, new Page.NavigateOptions().setWaitUntil(WaitUntilState.NETWORKIDLE));
                
                // 检查是否重定向到了SSO登录页面
                String currentUrl = page.url();
                log.info("当前页面URL: {}", currentUrl);
                
                if (currentUrl.startsWith(SSO_LOGIN_URL)) {
                    String errorMsg = "用户未登录或会话已过期，请先使用HisenseSsoLoginTool进行登录";
                    log.error(errorMsg);
                    return errorMsg;
                }
                
                
                

                // 等待页面完全加载完成
                page.waitForLoadState(LoadState.NETWORKIDLE);
                
                // 等待关键元素加载完成，确保页面完全就绪
                try {
                    page.waitForSelector("input[type='radio'][name*='oprGroup'], input[type='radio'][name*='fdNotifyLevel']", 
                        new Page.WaitForSelectorOptions().setState(com.microsoft.playwright.options.WaitForSelectorState.VISIBLE).setTimeout(5000));
                } catch (com.microsoft.playwright.TimeoutError e) {
                    log.warn("关键审批元素未在预期时间内加载完成，继续执行审批操作");
                }
                
                // 执行审批操作
                performApprovalOperation(page, approvalOpinion);
                
                // 截图并保存
                takeScreenshotAndSave(page, "lbpm_approval_success_" + ssoUsername);
                
                log.info("请假审批处理完成");
                
                return "请假审批处理成功";
            } catch (Exception e) {
                String errorMsg = "请假审批处理失败: " + e.getMessage();
                log.error("请假审批处理失败", e);
                
                // 如果页面对象存在，截图保存错误页面
                if (page != null) {
                    try {
                        takeScreenshotAndSave(page, "lbpm_approval_fail_" + ssoUsername);
                    } catch (Exception screenshotException) {
                        log.warn("截图保存失败: {}", screenshotException.getMessage());
                    }
                }
                
                return errorMsg;
            } finally {
                // 不立即关闭页面，让服务器完成审批处理
                // 保留页面、上下文和浏览器实例供后续操作使用
                // 仅在发生异常时才关闭页面
                if (page != null) {
                    try {
                        // 可以选择保留页面不关闭，或者等待一段时间后关闭
                        // 目前保持原逻辑，但实际使用中可能需要根据业务需求调整
                        log.debug("保留页面实例以等待服务器完成审批处理");
                    } catch (Exception e) {
                        log.warn("处理页面实例时发生异常: {}", e.getMessage());
                    }
                }
            }
        });
    }

    /**
     * 工具方法：获取海信业务系统的网页内容（自动处理SSO认证）
     * 
     * @param businessSystemUrl 海信业务系统页面URL
     * @return 页面内容（HTML文本）
     */
    @Tool(description = "获取海信LBPM业务系统的网页内容，需要先使用HisenseSsoLoginTool登录")
    public String getHisenseLbpmBusinessSystemContent(String businessSystemUrl) {
        Map<String, Object> params = new HashMap<>();
        params.put("businessSystemUrl", businessSystemUrl);
        
        return execute("getHisenseLbpmBusinessSystemContent", params, () -> {
            String ssoUsername = getSsoUsername();
            log.info("开始为用户 {} 获取海信业务系统内容，URL: {}", ssoUsername, businessSystemUrl);
            
            // 参数校验
            if (ssoUsername == null || ssoUsername.isEmpty()) {
                String errorMsg = "用户ID不能为空";
                log.error(errorMsg);
                return errorMsg;
            }
            
            if (businessSystemUrl == null || businessSystemUrl.isEmpty()) {
                String errorMsg = "业务系统URL不能为空";
                log.error(errorMsg);
                return errorMsg;
            }

            Page page = null;
            
            try {
                // 获取用户专用的浏览器上下文
                BrowserContext userContext = playwrightManager.getUserContext(ssoUsername);
                
                // 创建新页面
                page = userContext.newPage();
                
                // 访问业务系统页面
                log.info("正在访问业务系统页面: {}", businessSystemUrl);
                page.navigate(businessSystemUrl, new Page.NavigateOptions().setWaitUntil(WaitUntilState.NETWORKIDLE));
                
                // 检查是否重定向到了SSO登录页面
                String currentUrl = page.url();
                log.info("当前页面URL: {}", currentUrl);
                
                if (currentUrl.startsWith(SSO_LOGIN_URL)) {
                    String errorMsg = "用户未登录或会话已过期，请先使用HisenseSsoLoginTool进行登录";
                    log.error(errorMsg);
                    return errorMsg;
                }
                
                // 提取页面内容
                String content = page.locator("body").innerText();
                log.info("成功获取业务系统页面内容");
                
                // 检查是否包含错误信息
                if (content.contains("InvalidStateError") && content.contains("setRequestHeader")) {
                    log.warn("检测到页面中可能存在JavaScript错误，但这不会影响主要功能");
                }
                
                return content;
            } catch (Exception e) {
                String errorMsg = "获取海信业务系统内容失败: " + e.getMessage();
                log.error("获取海信业务系统内容失败", e);
                return errorMsg;
            } finally {
                // 释放页面资源，但保留浏览器上下文供后续使用
                if (page != null) {
                    try {
                        page.close();
                    } catch (Exception e) {
                        log.warn("关闭页面时发生异常: {}", e.getMessage());
                    }
                }
            }
        });
    }

    /**
     * 执行审批操作
     * 
     * @param page 当前页面对象
     * @param approvalOpinion 审批意见
     * @throws Exception 审批过程中的异常
     */
    private void performApprovalOperation(Page page, String approvalOpinion) throws Exception {
        log.info("开始执行审批操作");
        
        try {
            // 定位审批操作单选框 - 尝试新的选择器
            String operationRadioSelector = "input[type='radio'][name='sysWfBusinessForm.fdNotifyLevel'][value='1']";
            log.debug("正在定位审批操作单选框: {}", operationRadioSelector);
            Locator operationRadio = page.locator(operationRadioSelector);
            if (operationRadio.count() > 0) {
                operationRadio.click();
                log.debug("审批操作单选框选择完成 (使用新选择器)");
            } else {
                // 如果新选择器未找到元素，尝试原选择器
                operationRadioSelector = "input[type='radio'][alerttext=''][key='operationType'][name='oprGroup'][value='handler_pass:通过']";
                log.debug("新选择器未找到元素，尝试原选择器: {}", operationRadioSelector);
                operationRadio = page.locator(operationRadioSelector);
                if (operationRadio.count() > 0) {
                    operationRadio.click();
                    log.debug("审批操作单选框选择完成 (使用原选择器)");
                } else {
                    throw new RuntimeException("未找到审批操作单选框");
                }
            }

            // 定位审批意见输入框并填入内容 - 尝试新的选择器
            String opinionTextareaSelector = "textarea[name='fdUsageContent'][class='process_review_content'][key='auditNode'][subject='处理意见']";
            log.debug("正在定位审批意见输入框: {}", opinionTextareaSelector);
            Locator opinionTextarea = page.locator(opinionTextareaSelector);
            if (opinionTextarea.count() > 0) {
                opinionTextarea.fill(approvalOpinion);
                log.debug("审批意见输入完成 (使用新选择器)");
            } else {
                // 如果新选择器未找到元素，尝试原选择器
                opinionTextareaSelector = "textarea[name='fdUsageContent'][class='process_review_content'][key='auditNode']";
                log.debug("新选择器未找到元素，尝试原选择器: {}", opinionTextareaSelector);
                opinionTextarea = page.locator(opinionTextareaSelector);
                if (opinionTextarea.count() > 0) {
                    opinionTextarea.fill(approvalOpinion);
                    log.debug("审批意见输入完成 (使用原选择器)");
                } else {
                    throw new RuntimeException("未找到审批意见输入框");
                }
            }

            // 定位并点击提交按钮 - 尝试新的选择器
            String submitButtonSelector = "input[id='process_review_button'][class='process_review_button'][type='button'][value='提交']";
            log.debug("正在定位提交按钮: {}", submitButtonSelector);
            Locator submitButton = page.locator(submitButtonSelector);
            if (submitButton.count() > 0) {
                submitButton.click();
                log.info("提交按钮点击完成 (使用新选择器)");
            } else {
                // 如果新选择器未找到元素，尝试原选择器
                submitButtonSelector = "input[id='process_review_button'][class='process_review_button'][type='button'][value='提交']";
                log.debug("新选择器未找到元素，尝试原选择器: {}", submitButtonSelector);
                submitButton = page.locator(submitButtonSelector);
                if (submitButton.count() > 0) {
                    submitButton.click();
                    log.info("提交按钮点击完成 (使用原选择器)");
                } else {
                    throw new RuntimeException("未找到提交按钮");
                }
            }
            
            // 等待提交完成
            page.waitForLoadState(LoadState.NETWORKIDLE);
            
            // 提交后等待服务器处理完成审批，检查是否有成功提示或页面跳转
            try {
                // 等待可能的成功消息提示
                page.waitForSelector("text=您的操作已成功", new Page.WaitForSelectorOptions().setState(com.microsoft.playwright.options.WaitForSelectorState.VISIBLE).setTimeout(10000));
                log.info("检测到审批提交成功提示");
            } catch (com.microsoft.playwright.TimeoutError e) {
                log.info("在预期时间内未检测到提交成功提示，继续等待");
            } catch (Exception e) {
                log.warn("等待成功提示时发生异常，继续执行: {}", e.getMessage());
            }
            
            // 额外等待一段时间确保服务器处理完成
            page.waitForTimeout(5000); // 等待5秒让服务器完成处理
            log.info("审批操作执行完成，已等待服务器响应");
        } catch (Exception e) {
            log.error("审批操作过程中发生异常", e);
            throw new RuntimeException("审批操作失败: " + e.getMessage(), e);
        }
    }

    /**
     * 截图并保存到存储目录
     * 
     * @param page 当前页面对象
     * @param fileName 文件名前缀
     */
    private void takeScreenshotAndSave(Page page, String fileName) {
        try {
            // 确保存储目录存在
            File storageDir = new File(STORAGE_DIR);
            if (!storageDir.exists()) {
                storageDir.mkdirs();
            }
            
            // 生成带时间戳的文件名
            String timestamp = LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyyMMdd_HHmmss"));
            String fullFileName = String.format("%s_%s.png", fileName, timestamp);
            String filePath = Paths.get(STORAGE_DIR, fullFileName).toString();
            
            // 截图并保存
            page.screenshot(new Page.ScreenshotOptions().setPath(Paths.get(filePath)));
            log.info("截图已保存至: {}", filePath);
        } catch (Exception e) {
            log.error("截图保存失败: {}", e.getMessage(), e);
        }
    }

    /**
     * 工具方法：自动查找并处理所有待审批的请假流程
     * 
     * @param approvalOpinion 审批意见
     * @return 处理结果
     */
    @Tool(description = "自动查找所有待审批的请假流程的网址，需要先使用HisenseSsoLoginTool登录")
    public List<String> processAllPendingLeaveApprovals() {
        Map<String, Object> params = new HashMap<>();
        
        return execute("processAllPendingLeaveApprovals", params, () -> {
            String ssoUsername = getSsoUsername();
            log.info("开始为用户 {} 处理所有待审批的请假流程", ssoUsername);
            
            // 参数校验
            if (ssoUsername == null || ssoUsername.isEmpty()) {
                String errorMsg = "用户ID不能为空";
                log.error(errorMsg);
                return List.of(errorMsg);
            }
            
            Page page = null;
            
            try {
                // 获取用户专用的浏览器上下文
                BrowserContext userContext = playwrightManager.getUserContext(ssoUsername);
                
                // 创建新页面
                page = userContext.newPage();
                
                // 访问待审批列表页面
                String approvalListUrl = "https://lbpm.hisense.com/km/review/?categoryId=1843775ea85be87f9756e2540e5b20b0&nodeType=CATEGORY#j_path=%2FlistAll&mydoc=all&cri.q=docStatus%3A20%3BfdTemplate%3A1843775ea85be87f9756e2540e5b20b0";
                log.info("正在访问待审批列表页面: {}", approvalListUrl);
                page.navigate(approvalListUrl, new Page.NavigateOptions().setWaitUntil(WaitUntilState.NETWORKIDLE));
                
                // 检查是否重定向到了SSO登录页面
                String currentUrl = page.url();
                log.info("当前页面URL: {}", currentUrl);
                
                if (currentUrl.startsWith(SSO_LOGIN_URL)) {
                    String errorMsg = "用户未登录或会话已过期，请先使用HisenseSsoLoginTool进行登录";
                    log.error(errorMsg);
                    return List.of(errorMsg);
                }
                
                // 等待页面完全加载完成
                page.waitForLoadState(LoadState.NETWORKIDLE);
                
                // 等待待审批项目加载完成
                try {
                    page.waitForSelector("span.com_subject", 
                        new Page.WaitForSelectorOptions().setState(com.microsoft.playwright.options.WaitForSelectorState.VISIBLE).setTimeout(10000));
                } catch (com.microsoft.playwright.TimeoutError e) {
                    log.info("在预期时间内未找到待审批项目，可能没有待审批的项目");
                    return List.of("没有找到待审批的项目");
                }
                
                ArrayList<String> urls= new ArrayList<>();
                
                // 查找所有待审批项目行，这些行包含kmss_fdid属性
                Locator approvalRows = page.locator("tr[kmss_fdid]");
                int itemCount = approvalRows.count();
                
                if (itemCount == 0) {
                    log.info("没有更多待审批项目，处理完成");
                    return List.of("没有更多待审批项目，处理完成");
                }
                
                log.info("找到 {} 个待审批项目", itemCount);
                
                // 遍历所有待审批项目行，提取kmss_fdid属性值并构建审批URL
                for (int i = 0; i < itemCount; i++) {
                    Locator currentApprovalRow = approvalRows.nth(i);
                    
                    // 获取kmss_fdid属性值
                    String kmssFdid = currentApprovalRow.getAttribute("kmss_fdid");
                    if (kmssFdid != null && !kmssFdid.isEmpty()) {
                        // 构建完整的审批URL
                        String approvalUrl = "https://lbpm.hisense.com/km/review/km_review_main/kmReviewMain.do?method=view&fdId=" + kmssFdid;
                        
                        // 获取审批项目文本（从span.com_subject中获取）
                        Locator approvalSubject = currentApprovalRow.locator("span.com_subject");
                        String approvalText = approvalSubject.count() > 0 ? approvalSubject.textContent() : "未知审批项目";
                        
                        log.info("获取到待审批项目: {}，链接: {}", approvalText, approvalUrl);
                        urls.add(approvalUrl);
                    } 
                }
                
                log.info("待审批处理完成");
                
                return urls;
            } catch (Exception e) {
                String errorMsg = "处理待审批项目失败: " + e.getMessage();
                log.error("处理待审批项目失败", e);
                
                // 如果页面对象存在，截图保存错误页面
                if (page != null) {
                    try {
                        takeScreenshotAndSave(page, "lbpm_pending_approval_fail_" + ssoUsername);
                    } catch (Exception screenshotException) {
                        log.warn("截图保存失败: {}", screenshotException.getMessage());
                    }
                }
                
                return List.of(errorMsg);
            } finally {
                // 不立即关闭页面，让服务器完成审批处理
                // 保留页面、上下文和浏览器实例供后续操作使用
                // 仅在发生异常时才关闭页面
                if (page != null) {
                    try {
                        log.debug("保留页面实例以等待服务器完成审批处理");
                    } catch (Exception e) {
                        log.warn("处理页面实例时发生异常: {}", e.getMessage());
                    }
                }
            }
        });
    }
}
