package pangea.hiagent.agent.data;

import org.springframework.beans.factory.annotation.Autowired;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.util.Map;
import java.util.List;
import java.util.HashMap;

/**
 * 从Agent实现示例 - 数据库查询Agent
 */
@Slf4j
@ToolTag("DB_QUERY")
@Component
public class DbQuerySlaveAgent implements SlaveAgent {
    
    @Autowired
    private AgentTaskStatusRepository taskStatusRepository;
    @Override
    public AgentResult execute(SubTask subTask) {
        try {
            Map<String, Object> params = subTask.getParams();
            log.info("执行数据库查询任务，参数: {}", params);
            
            // 从参数中获取查询类型和条件
            String queryType = (String) params.getOrDefault("queryType", "SELECT");
            String tableName = (String) params.get("tableName");
            Map<String, Object> conditions = (Map<String, Object>) params.get("conditions");
            
            // 根据查询类型执行不同的查询
            Object result = null;
            switch (queryType.toUpperCase()) {
                case "SELECT":
                    result = executeSelectQuery(tableName, conditions);
                    break;
                case "COUNT":
                    result = executeCountQuery(tableName, conditions);
                    break;
                case "EXISTS":
                    result = executeExistsQuery(tableName, conditions);
                    break;
                default:
                    throw new IllegalArgumentException("不支持的查询类型: " + queryType);
            }
            
            AgentResult agentResult = new AgentResult();
            agentResult.setData(result);
            agentResult.setSuccess(true);
            agentResult.setResultHash(HashCalculator.calculateHash(result != null ? result.toString() : ""));
            return agentResult;
            
        } catch (Exception e) {
            log.error("数据库查询执行失败: {}", e.getMessage(), e);
            return new AgentResult(false, "数据库查询失败: " + e.getMessage(), null);
        }
    }
    
    /**
     * 执行SELECT查询
     */
    private Object executeSelectQuery(String tableName, Map<String, Object> conditions) {
        if (tableName == null) {
            throw new IllegalArgumentException("表名不能为空");
        }
        
        // 根据表名选择对应的Repository进行查询
        if ("agent_task_status".equalsIgnoreCase(tableName)) {
            return executeAgentTaskStatusQuery(conditions);
        } else {
            // 默认返回模拟数据，实际应用中可以扩展更多表的查询
            Map<String, Object> mockResult = new HashMap<>();
            mockResult.put("message", "查询了表: " + tableName);
            mockResult.put("conditions", conditions);
            return mockResult;
        }
    }
    
    /**
     * 执行任务状态查询
     */
    private Object executeAgentTaskStatusQuery(Map<String, Object> conditions) {
        String taskId = (String) (conditions != null ? conditions.get("taskId") : null);
        String status = (String) (conditions != null ? conditions.get("status") : null);
        
        if (taskId != null) {
            // 根据任务ID查询
            AgentTaskStatus taskStatus = taskStatusRepository.selectById(taskId);
            return taskStatus != null ? taskStatus : new HashMap<String, Object>();
        } else if (status != null) {
            // 根据状态查询
            return taskStatusRepository.findByStatus(status);
        } else {
            // 查询所有任务状态
            return taskStatusRepository.selectList(null);
        }
    }
    
    /**
     * 执行COUNT查询
     */
    private Object executeCountQuery(String tableName, Map<String, Object> conditions) {
        if (tableName == null) {
            throw new IllegalArgumentException("表名不能为空");
        }
        
        if ("agent_task_status".equalsIgnoreCase(tableName)) {
            // 根据状态计数
            String status = (String) (conditions != null ? conditions.get("status") : null);
            if (status != null) {
                List<AgentTaskStatus> statuses = taskStatusRepository.findByStatus(status);
                return statuses.size();
            } else {
                // 返回总记录数
                return taskStatusRepository.selectCount(null);
            }
        } else {
            // 默认返回模拟计数
            return 0;
        }
    }
    
    /**
     * 执行EXISTS查询
     */
    private Object executeExistsQuery(String tableName, Map<String, Object> conditions) {
        if (tableName == null) {
            throw new IllegalArgumentException("表名不能为空");
        }
        
        if ("agent_task_status".equalsIgnoreCase(tableName)) {
            String taskId = (String) (conditions != null ? conditions.get("taskId") : null);
            if (taskId != null) {
                AgentTaskStatus status = taskStatusRepository.selectById(taskId);
                return status != null;
            }
        }
        
        // 默认返回false
        return false;
    }
}