package pangea.hiagent.agent;

import org.junit.jupiter.api.Test;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.test.context.TestPropertySource;

import pangea.hiagent.agent.data.*;

import static org.junit.jupiter.api.Assertions.*;

import java.util.Map;

import lombok.extern.slf4j.Slf4j;

@Slf4j
@SpringBootTest
@TestPropertySource(properties = {
    "agent.comm.use-remote=false",
    "resilience4j.circuitbreaker.instances.DbQuerySlaveAgent.sliding-window-size=10",
    "resilience4j.circuitbreaker.instances.DbQuerySlaveAgent.failure-rate-threshold=50",
    "resilience4j.circuitbreaker.instances.DbQuerySlaveAgent.wait-duration-in-open-state=10s",
    "resilience4j.retry.instances.DbQuerySlaveAgent.max-attempts=3",
    "resilience4j.retry.instances.DbQuerySlaveAgent.wait-duration=1s"
})
public class SimpleMultiAgentTest {
    
    @Test
    public void testSimpleAgentExecution() {
        log.info("执行简单Agent测试");
        
        // 创建子任务
        SubTask subTask = new SubTask();
        subTask.setSubTaskId("simple-test-task");
        subTask.setTaskId("main-test-task");
        subTask.setToolTag("DB_QUERY");
        subTask.setParams(Map.of("query", "SELECT * FROM test"));
        
        // 创建从Agent
        SlaveAgent dbAgent = new DbQuerySlaveAgent();
        
        // 执行任务
        AgentResult result = dbAgent.execute(subTask);
        
        // 验证结果
        assertNotNull(result);
        assertTrue(result.isSuccess());
        log.info("简单Agent测试完成");
    }
    
    @Test
    public void testAgentChainSimple() {
        log.info("执行简单能力链测试");
        
        // 创建能力链
        AgentChain chain = new AgentChain();
        SlaveAgent dbAgent = new DbQuerySlaveAgent();
        chain.addAgent(dbAgent);
        
        // 创建子任务
        SubTask subTask = new SubTask();
        subTask.setSubTaskId("chain-test-task");
        subTask.setTaskId("main-test-task");
        subTask.setToolTag("DB_QUERY");
        subTask.setParams(Map.of("query", "SELECT COUNT(*) FROM test"));
        
        // 执行链
        AgentResult result = chain.execute(subTask);
        
        // 验证结果
        assertNotNull(result);
        assertTrue(result.isSuccess());
        log.info("简单能力链测试完成");
    }
    
    @Test
    public void testCooperateCommand() {
        log.info("执行协同命令测试");
        
        // 创建协同命令
        CooperateCommand command = new CooperateCommand();
        command.setAction("TEST_ACTION");
        command.setTaskId("test-task-id");
        command.setSenderAgentId("test-sender");
        command.setReceiverAgentId("test-receiver");
        
        // 验证命令属性
        assertNotNull(command.getMessageId());
        assertEquals("TEST_ACTION", command.getAction());
        assertEquals("test-task-id", command.getTaskId());
        assertEquals("test-sender", command.getSenderAgentId());
        assertEquals("test-receiver", command.getReceiverAgentId());
        
        log.info("协同命令测试完成");
    }
}
