package pangea.hiagent.agent;

import org.junit.jupiter.api.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.test.context.TestPropertySource;

import pangea.hiagent.agent.data.*;

import static org.junit.jupiter.api.Assertions.*;

import java.util.List;
import java.util.Map;

import lombok.extern.slf4j.Slf4j;

@Slf4j
@SpringBootTest
@TestPropertySource(properties = {
    "agent.comm.use-remote=false",
    "agent.gray.ratio=10",
    "agent.gray.white-list=test_task_001"
})
class MultiAgentCollaborationTest {

    @Test
    void testAgentChainExecution() {
        // 创建子任务
        SubTask subTask = new SubTask();
        subTask.setSubTaskId("test-subtask-1");
        subTask.setTaskId("test-task-1");
        subTask.setToolTag("DB_QUERY");
        subTask.setParams(Map.of("query", "SELECT * FROM users"));

        // 创建从Agent
        SlaveAgent dbAgent = new DbQuerySlaveAgent();
        
        // 创建能力链
        AgentChain chain = new AgentChain();
        chain.addAgent(dbAgent);
        
        // 执行链
        AgentResult result = chain.execute(subTask);
        
        // 验证结果
        assertNotNull(result);
        assertTrue(result.isSuccess());
        assertNotNull(result.getData());
    }

    @Test
    void testTaskDependencyResolution() {
        // 创建一些测试子任务
        SubTask task1 = new SubTask();
        task1.setSubTaskId("task1");
        task1.setDependOn(List.of()); // 无依赖
        
        SubTask task2 = new SubTask();
        task2.setSubTaskId("task2");
        task2.setDependOn(List.of("task1")); // 依赖task1
        
        SubTask task3 = new SubTask();
        task3.setSubTaskId("task3");
        task3.setDependOn(List.of("task1")); // 依赖task1
        
        SubTask task4 = new SubTask();
        task4.setSubTaskId("task4");
        task4.setDependOn(List.of("task2", "task3")); // 依赖task2和task3

        List<SubTask> subTasks = List.of(task1, task2, task3, task4);
        
        TaskDependencyResolver resolver = new TaskDependencyResolver();
        List<List<SubTask>> batches = resolver.resolve(subTasks);
        
        // 验证批次解析
        assertEquals(3, batches.size(), "应该有3个批次");
        assertEquals(1, batches.get(0).size(), "第一批应该有1个任务(task1)");
        assertEquals(2, batches.get(1).size(), "第二批应该有2个任务(task2, task3)");
        assertEquals(1, batches.get(2).size(), "第三批应该有1个任务(task4)");
        
        // 验证第一批是task1
        assertEquals("task1", batches.get(0).get(0).getSubTaskId());
        
        // 验证第二批包含task2和task3
        List<String> secondBatchIds = List.of(
            batches.get(1).get(0).getSubTaskId(),
            batches.get(1).get(1).getSubTaskId()
        );
        assertTrue(secondBatchIds.contains("task2"));
        assertTrue(secondBatchIds.contains("task3"));
        
        // 验证第三批是task4
        assertEquals("task4", batches.get(2).get(0).getSubTaskId());
    }

    @Test
    void testCooperateCommandCreation() {
        CooperateCommand command = new CooperateCommand();
        
        assertNotNull(command.getMessageId());
        assertTrue(command.getMessageId().length() > 0);
        
        command.setAction("TASK_ASSIGN");
        command.setTaskId("test-task");
        command.setSenderAgentId("master-agent");
        command.setReceiverAgentId("slave-agent");
        
        assertEquals("TASK_ASSIGN", command.getAction());
        assertEquals("test-task", command.getTaskId());
        assertEquals("master-agent", command.getSenderAgentId());
        assertEquals("slave-agent", command.getReceiverAgentId());
    }
    
    @Autowired
    private ModuleTestService moduleTestService;
    
    @Autowired
    private ModuleProperties moduleProperties;
    
    @Test
    void testModuleConfiguration() {
        // 测试模块属性配置
        assertNotNull(moduleProperties);
        assertTrue(moduleProperties.isEnabled());
        assertTrue(moduleProperties.getLog().isEnabled());
        assertTrue(moduleProperties.getCheck().isEnabled());
        assertTrue(moduleProperties.getMonitor().isEnabled());
        assertTrue(moduleProperties.getFallback().isEnabled());
        
        // 测试模块功能
        moduleTestService.testAllModules();
    }
    
    @Test
    void testModuleHotReload() {
        // 测试热更新功能
        moduleTestService.testHotReload();
        
        // 验证配置属性支持刷新
        assertNotNull(moduleProperties.getLog());
        assertNotNull(moduleProperties.getCheck());
        assertNotNull(moduleProperties.getMonitor());
        assertNotNull(moduleProperties.getFallback());
        assertNotNull(moduleProperties.getCache());
        assertNotNull(moduleProperties.getScheduler());
        assertNotNull(moduleProperties.getPerformance());
        assertNotNull(moduleProperties.getSecurity());
    }
    
    @Test
    void testExtendedModules() {
        // 测试缓存管理模块
        CacheManagementModule cacheModule = new CacheManagementModule();
        cacheModule.initializeCache();
        
        // 测试任务调度模块
        TaskSchedulerModule schedulerModule = new TaskSchedulerModule();
        assertEquals(0, schedulerModule.getQueueSize());
        
        // 测试性能监控模块
        PerformanceMonitorModule perfModule = new PerformanceMonitorModule();
        assertTrue(perfModule.getSuccessRate() >= 0.0);
        
        // 测试安全审计模块
        SecurityAuditModule securityModule = new SecurityAuditModule();
        assertTrue(securityModule.getSecurityEventStats().size() >= 0);
        
        System.out.println("扩展模块测试完成");
    }
}