package pangea.hiagent.agent.data;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Service;

import jakarta.annotation.PostConstruct;
import java.util.List;
import java.util.concurrent.CompletableFuture;

/**
 * 任务状态服务，处理故障自愈逻辑
 */
@Slf4j
@Service
public class TaskStatusService {
    
    @Autowired
    private AgentTaskStatusRepository taskStatusRepository;
    
    @Autowired
    private MasterAgent masterAgent;
    
    /**
     * 超时兜底：主Agent通过定时任务扫描RUNNING状态且超过超时时间的任务，
     * 自动触发重试或标记为失败
     */
    @Scheduled(fixedRate = 5000) // 每5秒检查一次
    public void checkTimeoutTasks() {
        List<String> timeoutTaskIds = taskStatusRepository.findTimeoutTasks(System.currentTimeMillis());
        for (String taskId : timeoutTaskIds) {
            log.info("发现超时任务，开始重试: {}", taskId);
            retryTask(taskId);
        }
    }
    
    /**
     * 崩溃恢复：应用重启后，主Agent自动扫描RUNNING状态的任务，
     * 基于数据库中存储的状态数据恢复执行
     */
    @PostConstruct
    public void recoverUnfinishedTasks() {
        List<AgentTaskStatus> runningTasks = taskStatusRepository.findByStatus("RUNNING");
        log.info("应用启动，发现{}个未完成的任务，开始恢复执行", runningTasks.size());
        for (AgentTaskStatus status : runningTasks) {
            MainTask task = rebuildTaskFromStatus(status);
            CompletableFuture.runAsync(() -> {
                try {
                    masterAgent.executeMainTask(task);
                } catch (Exception e) {
                    log.error("恢复任务执行失败: {}", status.getTaskId(), e);
                }
            });
        }
    }
    
    /**
     * 重试任务
     */
    private void retryTask(String taskId) {
        try {
            // 更新重试次数
            AgentTaskStatus status = taskStatusRepository.selectById(taskId);
            if (status != null) {
                int newRetryCount = (status.getRetryCount() != null ? status.getRetryCount() : 0) + 1;
                status.setRetryCount(newRetryCount);
                
                // 如果重试次数超过阈值，标记为失败
                if (newRetryCount > 3) { // 最多重试3次
                    status.setStatus("FAIL");
                    taskStatusRepository.updateById(status);
                    log.warn("任务{}重试次数超过阈值，标记为失败", taskId);
                    return;
                }
                
                // 更新状态并重新执行任务
                status.setStatus("READY");
                taskStatusRepository.updateById(status);
                
                // 重新执行任务逻辑（这里需要根据实际情况实现）
                log.info("任务{}重试，当前重试次数: {}", taskId, newRetryCount);
            }
        } catch (Exception e) {
            log.error("重试任务失败: {}", taskId, e);
        }
    }
    
    /**
     * 从状态数据重建任务
     */
    private MainTask rebuildTaskFromStatus(AgentTaskStatus status) {
        MainTask task = new MainTask();
        task.setTaskId(status.getTaskId());
        
        // 从result字段中反序列化出原始任务数据
        if (status.getResult() != null && !status.getResult().isEmpty()) {
            try {
                // 使用Jackson ObjectMapper进行JSON反序列化
                com.fasterxml.jackson.databind.ObjectMapper objectMapper = new com.fasterxml.jackson.databind.ObjectMapper();
                objectMapper.findAndRegisterModules(); // 注册JavaTimeModule等模块
                
                // 尝试将result字段解析为MainTask对象
                MainTask deserializedTask = objectMapper.readValue(status.getResult(), MainTask.class);
                
                // 使用反序列化的数据更新任务，保留ID
                task.setTaskName(deserializedTask.getTaskName());
                task.setSubTasks(deserializedTask.getSubTasks());
                task.setExpectedHash(deserializedTask.getExpectedHash());
                task.setMasterAgentId(deserializedTask.getMasterAgentId());
                
                log.debug("从状态数据成功重建任务: {}", status.getTaskId());
            } catch (Exception e) {
                log.error("从状态数据重建任务失败: taskId={}, result={}, error={}", 
                         status.getTaskId(), status.getResult(), e.getMessage());
                
                // 如果反序列化失败，使用默认值
                task.setTaskName("Recovered Task - " + status.getTaskId());
                task.setSubTasks(List.of());
                task.setExpectedHash(status.getResultHash());
                task.setMasterAgentId(null);
            }
        } else {
            // 如果没有result数据，使用基本任务信息
            task.setTaskName("Recovered Task - " + status.getTaskId());
            task.setSubTasks(List.of());
            task.setExpectedHash(status.getResultHash());
            task.setMasterAgentId(null);
            
            log.warn("任务状态中无结果数据，使用默认值重建任务: {}", status.getTaskId());
        }
        
        return task;
    }
}