package pangea.hiagent.agent.react;

import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import lombok.extern.slf4j.Slf4j;
import pangea.hiagent.agent.service.UserSseService;
import pangea.hiagent.common.utils.UserUtils;
import pangea.hiagent.agent.data.WorkPanelEvent;

/**
 * ReAct回调类
 */
@Slf4j
@Component
public class DefaultReactCallback implements ReactCallback {

    @Autowired
    private UserSseService userSseService;

    // EventSplitter functionality integrated directly
    private final List<String> keywords = Arrays.asList(
            "Thought", "Action", "Observation", "Final_Answer");
    private final Pattern keywordPattern = Pattern.compile(
            String.format("(?i)(Thought|Action|Observation|Final[ _]Answer):", String.join("|", keywords)),
            Pattern.CASE_INSENSITIVE);

    private String currentType = null;
    private StringBuilder currentContent = new StringBuilder();
    private StringBuilder buffer = new StringBuilder();

    private volatile int stepNumber = 0;

    private void onStep(ReactStep reactStep) {

        String reactStepName = reactStep.getStepType().name();

        try {
            userSseService.sendWorkPanelEvent(WorkPanelEvent.builder()
                    .type(reactStepName)
                    .content(reactStep.getContent())
                    .userId(UserUtils.getCurrentUserIdStatic())
                    .build());
        } catch (IOException e) {
            log.error("发送ReAct步骤到WorkPanel失败: 类型={}, 内容摘要={}",
                    reactStep.getStepType(),
                    reactStep.getContent() != null
                            ? reactStep.getContent().substring(0, Math.min(50, reactStep.getContent().length()))
                            : "null",
                    e);
        }

        // 记录最终答案到日志
        log.info("[WorkPanel] 记录{} {}", reactStepName,
                reactStep.getContent().substring(0, Math.min(100, reactStep.getContent().length())));

    }

    // 每收到一个token/字符，调用此方法
    @Override
    public void onToken(String token) {
        buffer.append(token);

        // log.debug("当前缓冲区: {}", buffer.toString());

        Matcher matcher = keywordPattern.matcher(buffer);
        while (matcher.find()) {
            log.debug("发现新事件关键词: {}", matcher.group(1));
            // 发现新事件
            if (currentType != null && currentContent.length() > 0) {
                // 实时输出已分割事件
                onStep(new ReactStep(stepNumber++, ReactStepType.fromString(currentType), currentContent.toString()));
            }
            // 更新事件类型
            currentType = matcher.group(1);
            currentContent.setLength(0);
            // 累积匹配位置后的内容
            currentContent.append(buffer.substring(matcher.end()));
            // 重置buffer为剩余内容
            buffer.setLength(0);
            buffer.append(currentContent);
            // 重新查找
            matcher = keywordPattern.matcher(buffer);
        }
        // 检查是否有部分关键词在buffer末尾
        if (buffer.length() > 0) {
            // 检查是否可能是关键词的一部分
            boolean isPartialKeyword = false;
            String bufferStr = buffer.toString();
            for (String keyword : keywords) {
                if (keyword.startsWith(bufferStr) || bufferStr.startsWith(keyword)) {
                    isPartialKeyword = true;
                    break;
                }
            }
            if (!isPartialKeyword) {
                // 不是部分关键词，添加到内容中
                currentContent.append(buffer);
                buffer.setLength(0);
            }
        }
    }

    // 流式结束时，调用此方法输出最后一个事件
    @Override
    public void endStream() {
        if (currentType != null && currentContent.length() > 0) {
            onStep(new ReactStep(stepNumber++, ReactStepType.fromString(currentType), currentContent.toString()));
        }
    }
}