package pangea.hiagent.agent.data;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import java.util.concurrent.ConcurrentLinkedQueue;

/**
 * 双模消息总线核心实现
 */
@Slf4j
@Component
public class DualModeMessageBus {
    private final ConcurrentLinkedQueue<CooperateCommand> localQueue = new ConcurrentLinkedQueue<>();
    private final IdempotentChecker idempotentChecker;
    private final CommandRouter commandRouter;

    /** 是否启用远程MQ通信，通过配置文件控制 */
    private final boolean useRemote;

    public DualModeMessageBus(IdempotentChecker idempotentChecker,
                              CommandRouter commandRouter,
                              @Value("${agent.comm.use-remote:false}") boolean useRemote) {
        this.idempotentChecker = idempotentChecker;
        this.commandRouter = commandRouter;
        this.useRemote = useRemote;
    }

    /** 发送协同指令 */
    public void send(CooperateCommand command) {
        // 如果启用远程通信，这里需要通过其他方式发送（如HTTP或WebSocket）
        // 为简化实现，当前只使用本地队列
        localQueue.offer(command);
    }

    /** 消费本地队列消息，定时轮询 */
    @Scheduled(fixedRate = 100)
    public void consumeLocalQueue() {
        while (!localQueue.isEmpty()) {
            CooperateCommand command = localQueue.poll();
            if (command != null) {
                handleCommand(command);
            }
        }
    }

    /** 消息处理核心逻辑，包含幂等校验 */
    private void handleCommand(CooperateCommand command) {
        if (!idempotentChecker.check(command.getMessageId())) {
            return;
        }
        commandRouter.route(command);
    }
}