package pangea.hiagent.web.controller;

import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.servlet.mvc.method.annotation.SseEmitter;

import pangea.hiagent.agent.service.AgentChatService;
import pangea.hiagent.web.dto.ChatRequest;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.Valid;
import jakarta.validation.constraints.NotBlank;

/**
 * Agent 对话控制器
 * 职责：接收HTTP请求并转发给AgentChatService处理
 */
@Slf4j
@RestController
@RequestMapping("/api/v1/agent")
public class AgentChatController {
    
    private final AgentChatService agentChatService;
    
    public AgentChatController(AgentChatService agentChatService) {
        this.agentChatService = agentChatService;
    }
    
    /**
     * 流式对话接口
     * 
     * @param agentId Agent ID
     * @param chatRequest 对话请求
     * @param response HTTP响应
     * @return SSE emitter
     */
    @PostMapping("/chat-stream")
    public SseEmitter chatStream(
            @RequestParam @NotBlank(message = "Agent ID不能为空") String agentId,
            @RequestBody @Valid ChatRequest chatRequest,
            HttpServletResponse response) {
        log.info("接收到流式对话请求，AgentId: {}", agentId);
        
        // 注意：权限检查已由 SseAuthorizationFilter 在更早的阶段处理
        // 此时响应尚未开始流式传输，确保在流式开始前完成所有权限验证
        // 这样可以避免在流式传输过程中突然抛出异常导致响应已提交的问题
        
        // 仅验证Agent存在性，权限检查由过滤器处理
        // 为避免安全异常，这里不直接调用agentService.getAgent()，而是让agentChatService处理
        
        // 调用异步处理
        return agentChatService.handleChatStream(agentId, chatRequest, response);
    }
}