package pangea.hiagent.agent.service;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import pangea.hiagent.agent.data.*;
import pangea.hiagent.model.Agent;

import java.util.*;
import java.util.regex.Pattern;

/**
 * 多Agent协同任务生成服务
 * 分析用户输入并判断是否需要启动多Agent协同任务
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class MultiAgentTaskGeneratorService {

    private final MasterAgent masterAgent;

    // 识别需要多Agent协同的关键词
    private static final Set<String> COLLABORATION_KEYWORDS = Set.of(
        "协作", "协同", "多个", "多步骤", "多阶段", "联合", "分工", "配合", "并行", "协调",
        "multi-agent", "collaboration", "cooperation", "multiple agents", "teamwork", "coordinate"
    );

    // 识别任务类型的关键词
    private static final Map<String, String> TASK_TYPE_KEYWORDS = Map.of(
        "查询", "search",
        "计算", "calculate", 
        "分析", "analyze",
        "生成", "generate",
        "处理", "process",
        "转换", "transform",
        "验证", "verify",
        "汇总", "summarize"
    );

    /**
     * 分析用户输入，判断是否需要启动多Agent协同任务
     * 
     * @param userInput 用户输入的文本
     * @return 是否需要启动多Agent协同任务
     */
    public boolean shouldStartCollaboration(String userInput) {
        if (userInput == null || userInput.trim().isEmpty()) {
            return false;
        }

        String lowerInput = userInput.toLowerCase();
        
        // 检查是否包含协作关键词
        for (String keyword : COLLABORATION_KEYWORDS) {
            if (lowerInput.contains(keyword.toLowerCase())) {
                log.info("检测到协作关键词，需要启动多Agent协同任务: {}", keyword);
                return true;
            }
        }

        // 检查是否包含多个任务或复杂任务的描述
        if (containsMultipleTasks(userInput)) {
            log.info("检测到多个任务描述，需要启动多Agent协同任务");
            return true;
        }

        // 检查是否包含复杂任务的描述
        if (isComplexTask(userInput)) {
            log.info("检测到复杂任务，可能需要多Agent协同");
            return true;
        }

        return false;
    }

    /**
     * 检查输入是否包含多个任务
     */
    private boolean containsMultipleTasks(String input) {
        // 检查是否有并列的动词或任务描述
        String[] taskIndicators = {"然后", "并且", "同时", "另外", "接着", "之后", "以及", "and", "then", "also", "plus"};
        
        for (String indicator : taskIndicators) {
            if (input.contains(indicator)) {
                return true;
            }
        }
        
        // 检查是否有分号或逗号分隔的多个任务
        String[] parts = input.split("[,;，；]");
        return parts.length > 2; // 如果有超过2个部分，可能包含多个任务
    }

    /**
     * 判断是否为复杂任务
     */
    private boolean isComplexTask(String input) {
        // 复杂任务通常包含多个步骤或需要多种能力
        String[] complexityIndicators = {"需要", "要求", "步骤", "流程", "过程", "方法", "方案", "策略", "需要先.*再", "先.*后"};
        
        for (String indicator : complexityIndicators) {
            if (Pattern.compile(indicator, Pattern.CASE_INSENSITIVE).matcher(input).find()) {
                return true;
            }
        }
        
        // 检查字符长度，如果较长可能包含复杂任务
        return input.length() > 50;
    }

    /**
     * 根据用户输入生成多Agent协同任务
     * 
     * @param userInput 用户输入的文本
     * @param agent 当前选中的Agent
     * @return 生成的多Agent协同任务
     */
    public MainTask generateCollaborationTask(String userInput, Agent agent) {
        MainTask mainTask = new MainTask();
        mainTask.setTaskId("collab_" + System.currentTimeMillis());
        mainTask.setTaskName("多Agent协同任务: " + userInput.substring(0, Math.min(30, userInput.length())));
        mainTask.setMasterAgentId(agent.getId());
        
        // 分析用户需求，生成子任务
        List<SubTask> subTasks = analyzeUserInput(userInput);
        mainTask.setSubTasks(subTasks);
        
        log.info("生成多Agent协同任务: {}，包含 {} 个子任务", mainTask.getTaskId(), subTasks.size());
        return mainTask;
    }

    /**
     * 分析用户输入，生成子任务列表
     */
    private List<SubTask> analyzeUserInput(String userInput) {
        List<SubTask> subTasks = new ArrayList<>();
        
        // 简单的自然语言分析，识别可能的任务类型
        String lowerInput = userInput.toLowerCase();
        
        // 根据关键词识别可能需要的工具类型
        List<String> requiredCapabilities = identifyCapabilities(userInput);
        
        if (requiredCapabilities.isEmpty()) {
            // 如果没有识别到特定能力，使用默认能力
            SubTask defaultTask = new SubTask();
            defaultTask.setSubTaskId("subtask_" + System.currentTimeMillis());
            defaultTask.setTaskId("default_task");
            defaultTask.setToolTag("GENERAL"); // 通用处理能力
            defaultTask.setParams(Map.of("input", userInput));
            subTasks.add(defaultTask);
        } else {
            // 为每种需要的能力创建一个子任务
            for (int i = 0; i < requiredCapabilities.size(); i++) {
                SubTask subTask = new SubTask();
                subTask.setSubTaskId("subtask_" + System.currentTimeMillis() + "_" + i);
                subTask.setTaskId("dynamic_task");
                subTask.setToolTag(requiredCapabilities.get(i));
                subTask.setParams(Map.of("input", userInput));
                
                // 设置任务依赖关系（简单的线性依赖）
                if (i > 0) {
                    subTask.setDependOn(List.of("subtask_" + System.currentTimeMillis() + "_" + (i - 1)));
                }
                
                subTasks.add(subTask);
            }
        }
        
        return subTasks;
    }

    /**
     * 识别用户需求中需要的Agent能力
     */
    private List<String> identifyCapabilities(String userInput) {
        List<String> capabilities = new ArrayList<>();
        String lowerInput = userInput.toLowerCase();
        
        // 检查是否存在数据库查询需求
        if (lowerInput.contains("查询") || lowerInput.contains("search") || 
            lowerInput.contains("database") || lowerInput.contains("db") ||
            lowerInput.contains("数据")) {
            capabilities.add("DB_QUERY");
        }
        
        // 检查是否存在计算需求
        if (lowerInput.contains("计算") || lowerInput.contains("calculate") ||
            lowerInput.contains("数学") || lowerInput.contains("算术") ||
            lowerInput.contains("sum") || lowerInput.contains("count")) {
            capabilities.add("CALCULATION");
        }
        
        // 检查是否存在分析需求
        if (lowerInput.contains("分析") || lowerInput.contains("analyze") ||
            lowerInput.contains("统计") || lowerInput.contains("统计") ||
            lowerInput.contains("统计")) {
            capabilities.add("ANALYSIS");
        }
        
        // 检查是否存在文档处理需求
        if (lowerInput.contains("文档") || lowerInput.contains("document") ||
            lowerInput.contains("文件") || lowerInput.contains("file") ||
            lowerInput.contains("读取") || lowerInput.contains("read")) {
            capabilities.add("DOCUMENT_PROCESSING");
        }
        
        // 检查是否存在网络请求需求
        if (lowerInput.contains("获取") || lowerInput.contains("fetch") ||
            lowerInput.contains("网络") || lowerInput.contains("web") ||
            lowerInput.contains("url") || lowerInput.contains("链接")) {
            capabilities.add("WEB_REQUEST");
        }
        
        // 如果没有识别到特定能力，但检测到复杂需求，则使用通用能力
        if (capabilities.isEmpty() && (userInput.length() > 30 || containsMultipleTasks(userInput))) {
            capabilities.add("GENERAL");
        }
        
        return capabilities;
    }

    /**
     * 获取系统中可用的从Agent列表
     */
    public List<String> getAvailableSlaveAgents() {
        return new ArrayList<>(masterAgent.getAgentMap().keySet());
    }
}