package pangea.hiagent.agent.data;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.context.event.EventListener;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 安全审计模块
 * 统一处理安全相关事件
 */
@Slf4j
@Component
@RefreshScope
@ConditionalOnProperty(prefix = "agent.module.security", name = "enabled", havingValue = "true", matchIfMissing = false)
public class SecurityAuditModule {

    @Autowired
    private ModuleProperties moduleProperties;

    // 安全事件记录
    private final Map<String, SecurityEventRecord> securityEvents = new ConcurrentHashMap<>();

    /**
     * 监听任务执行事件，进行安全审计
     */
    @Async
    @EventListener
    public void handleTaskEvent(AgentTaskEvent event) {
        if (moduleProperties.getSecurity().isLogTaskAccess()) {
            logSecurityEvent("TASK_ACCESS", event.getTaskId(), "Task " + event.getEventType());
        }
    }

    /**
     * 记录安全事件
     */
    private void logSecurityEvent(String eventType, String resource, String details) {
        SecurityEventRecord record = new SecurityEventRecord(
            eventType,
            resource,
            details,
            System.currentTimeMillis()
        );
        
        securityEvents.put(record.getTimestamp() + "_" + resource, record);
        
        if (moduleProperties.getSecurity().isLogEnabled()) {
            log.info("Security Audit - Type: {}, Resource: {}, Details: {}", 
                eventType, resource, details);
        }
    }

    /**
     * 获取安全事件统计
     */
    public Map<String, Long> getSecurityEventStats() {
        Map<String, Long> stats = new java.util.HashMap<>();
        
        for (SecurityEventRecord record : securityEvents.values()) {
            String eventType = record.getEventType();
            stats.put(eventType, stats.getOrDefault(eventType, 0L) + 1);
        }
        
        return stats;
    }

    /**
     * 清空安全事件记录
     */
    public void clearSecurityEvents() {
        securityEvents.clear();
        log.debug("清空安全审计事件记录");
    }

    /**
     * 获取最近的安全事件
     */
    public java.util.List<SecurityEventRecord> getRecentSecurityEvents(int limit) {
        return securityEvents.values().stream()
            .sorted((a, b) -> Long.compare(b.getTimestamp(), a.getTimestamp()))
            .limit(limit)
            .toList();
    }
}

/**
 * 安全事件记录类
 */
class SecurityEventRecord {
    private final String eventType;
    private final String resource;
    private final String details;
    private final long timestamp;

    public SecurityEventRecord(String eventType, String resource, String details, long timestamp) {
        this.eventType = eventType;
        this.resource = resource;
        this.details = details;
        this.timestamp = timestamp;
    }

    // Getters
    public String getEventType() { return eventType; }
    public String getResource() { return resource; }
    public String getDetails() { return details; }
    public long getTimestamp() { return timestamp; }
}

/**
 * 安全审计模块属性
 */
class SecurityAuditModuleProperties {
    private boolean enabled = true;
    private boolean logEnabled = true;
    private boolean logTaskAccess = true;
    private boolean logDataAccess = true;
    private String auditLogPath = "./audit";
    private int retentionDays = 30;

    // Getters and setters
    public boolean isEnabled() { return enabled; }
    public void setEnabled(boolean enabled) { this.enabled = enabled; }
    public boolean isLogEnabled() { return logEnabled; }
    public void setLogEnabled(boolean logEnabled) { this.logEnabled = logEnabled; }
    public boolean isLogTaskAccess() { return logTaskAccess; }
    public void setLogTaskAccess(boolean logTaskAccess) { this.logTaskAccess = logTaskAccess; }
    public boolean isLogDataAccess() { return logDataAccess; }
    public void setLogDataAccess(boolean logDataAccess) { this.logDataAccess = logDataAccess; }
    public String getAuditLogPath() { return auditLogPath; }
    public void setAuditLogPath(String auditLogPath) { this.auditLogPath = auditLogPath; }
    public int getRetentionDays() { return retentionDays; }
    public void setRetentionDays(int retentionDays) { this.retentionDays = retentionDays; }
}