package pangea.hiagent.web.controller;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;
import pangea.hiagent.agent.data.AgentTaskStatus;
import pangea.hiagent.web.dto.ApiResponse;
import pangea.hiagent.web.dto.PageData;
import pangea.hiagent.web.service.AgentService;
import pangea.hiagent.web.service.CollaborationService;
import pangea.hiagent.model.Agent;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 多Agent协作控制器
 * 提供协作任务管理功能
 */
@Slf4j
@RestController
@RequestMapping("/api/v1/agent/collaboration")
@RequiredArgsConstructor
public class CollaborationController {

    private final CollaborationService collaborationService;
    private final AgentService agentService;

    /**
     * 获取协同任务统计信息
     */
    @GetMapping("/stats")
    public ApiResponse<Map<String, Object>> getCollaborationStats() {
        try {
            log.info("获取协同任务统计信息");
            
            Map<String, Object> stats = collaborationService.getCollaborationStats();
            
            return ApiResponse.success(stats, "获取统计信息成功");
        } catch (Exception e) {
            log.error("获取协同任务统计信息失败", e);
            return ApiResponse.error(500, "获取统计信息失败: " + e.getMessage());
        }
    }

    /**
     * 获取协同任务列表
     */
    @GetMapping("/tasks")
    public ApiResponse<PageData<AgentTaskStatus>> getCollaborationTasks(
            @RequestParam(defaultValue = "1") Long page,
            @RequestParam(defaultValue = "10") Long size,
            @RequestParam(required = false) String status,
            @RequestParam(required = false) String keyword) {
        try {
            log.info("获取协同任务列表，页码: {}, 每页大小: {}, 状态: {}, 关键词: {}", page, size, status, keyword);
            
            // 获取过滤后的任务列表
            List<AgentTaskStatus> filteredTasks = collaborationService.getCollaborationTasks(status, keyword, Math.toIntExact(page), Math.toIntExact(size));
            
            // 获取总任务数
            int total = collaborationService.getTotalTaskCount(status, keyword);
            
            // 分页处理
            int start = Math.toIntExact((page - 1) * size);
            int end = Math.toIntExact(Math.min(page * size, filteredTasks.size()));
            
            if (start >= filteredTasks.size()) {
                start = Math.max(0, filteredTasks.size() - Math.toIntExact(size));
                end = filteredTasks.size();
            }
            
            List<AgentTaskStatus> pagedTasks = start < filteredTasks.size() ? 
                    filteredTasks.subList(start, end) : List.of();
            
            PageData<AgentTaskStatus> pageData = new PageData<>();
            pageData.setRecords(pagedTasks);
            pageData.setTotal((long) filteredTasks.size());
            pageData.setCurrent(page);
            pageData.setSize(size);
            pageData.setPages((long) Math.ceil((double) filteredTasks.size() / size));
            
            return ApiResponse.success(pageData, "获取任务列表成功");
        } catch (Exception e) {
            log.error("获取协同任务列表失败", e);
            return ApiResponse.error(500, "获取任务列表失败: " + e.getMessage());
        }
    }

    /**
     * 创建协同任务
     */
    @PostMapping("/tasks")
    public ApiResponse<Void> createCollaborationTask(@RequestBody Map<String, Object> taskParams) {
        try {
            log.info("创建协同任务: {}", taskParams);
            
            // 这里需要根据实际的多Agent协作实现来处理任务创建
            // 目前暂时返回成功响应
            String taskId = "task_" + System.currentTimeMillis();
            log.info("创建协同任务成功，任务ID: {}", taskId);
            
            return ApiResponse.success(null, "创建协同任务成功");
        } catch (Exception e) {
            log.error("创建协同任务失败", e);
            return ApiResponse.error(500, "创建协同任务失败: " + e.getMessage());
        }
    }

    /**
     * 重试任务
     */
    @PostMapping("/tasks/{taskId}/retry")
    public ApiResponse<Void> retryTask(@PathVariable String taskId) {
        try {
            log.info("重试任务: {}", taskId);
            
            collaborationService.retryTask(taskId);
            
            return ApiResponse.success(null, "任务重试成功");
        } catch (Exception e) {
            log.error("重试任务失败: {}", taskId, e);
            return ApiResponse.error(500, "重试任务失败: " + e.getMessage());
        }
    }

    /**
     * 获取协作相关的Agent统计信息
     */
    @GetMapping("/stats/agent")
    public ApiResponse<Map<String, Object>> getCollaborationAgentStats() {
        try {
            log.info("获取Agent统计信息");
            
            List<Agent> agents = agentService.listAgents();
            // 目前Agent模型中没有agentType字段，我们根据名称或其他特征来区分主从Agent
            // 这里暂时根据名称中是否包含"master"或"slave"来区分
            long masterAgents = agents.stream()
                    .filter(agent -> agent.getName() != null && 
                            (agent.getName().toLowerCase().contains("master") || 
                             agent.getName().toLowerCase().contains("主")))
                    .count();
            long slaveAgents = agents.stream()
                    .filter(agent -> agent.getName() != null && 
                            (agent.getName().toLowerCase().contains("slave") || 
                             agent.getName().toLowerCase().contains("从") ||
                             agent.getDescription() != null && 
                             agent.getDescription().toLowerCase().contains("slave")))
                    .count();
            long availableSlaveAgents = slaveAgents; // 假设所有从Agent都可用
            
            Map<String, Object> stats = Map.of(
                    "masterAgents", masterAgents,
                    "slaveAgents", slaveAgents,
                    "availableSlaveAgents", availableSlaveAgents
            );
            
            return ApiResponse.success(stats, "获取Agent统计信息成功");
        } catch (Exception e) {
            log.error("获取Agent统计信息失败", e);
            return ApiResponse.error(500, "获取Agent统计信息失败: " + e.getMessage());
        }
    }
    
    /**
     * 获取主Agent列表
     */
    @GetMapping("/master-agents")
    public ApiResponse<List<Agent>> getMasterAgents() {
        try {
            log.info("获取主Agent列表");
            
            List<Agent> allAgents = agentService.listAgents();
            List<Agent> masterAgents = allAgents.stream()
                    .filter(agent -> agent.getName() != null && 
                            (agent.getName().toLowerCase().contains("master") || 
                             agent.getName().toLowerCase().contains("主")))
                    .collect(Collectors.toList());
            
            return ApiResponse.success(masterAgents, "获取主Agent列表成功");
        } catch (Exception e) {
            log.error("获取主Agent列表失败", e);
            return ApiResponse.error(500, "获取主Agent列表失败: " + e.getMessage());
        }
    }
    
    /**
     * 获取从Agent列表
     */
    @GetMapping("/slave-agents")
    public ApiResponse<List<Agent>> getSlaveAgents() {
        try {
            log.info("获取从Agent列表");
            
            List<Agent> allAgents = agentService.listAgents();
            List<Agent> slaveAgents = allAgents.stream()
                    .filter(agent -> agent.getName() != null && 
                            (agent.getName().toLowerCase().contains("slave") || 
                             agent.getName().toLowerCase().contains("从") ||
                             agent.getDescription() != null && 
                             agent.getDescription().toLowerCase().contains("slave")))
                    .collect(Collectors.toList());
            
            return ApiResponse.success(slaveAgents, "获取从Agent列表成功");
        } catch (Exception e) {
            log.error("获取从Agent列表失败", e);
            return ApiResponse.error(500, "获取从Agent列表失败: " + e.getMessage());
        }
    }
}