package pangea.hiagent.web.controller;

import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import pangea.hiagent.rag.RagService;
import pangea.hiagent.web.dto.ApiResponse;
import pangea.hiagent.model.Agent;

import java.util.List;
import java.util.Map;

/**
 * RAG API控制器
 * 提供文档检索和RAG增强相关的API接口
 */
@Slf4j
@RestController
@RequestMapping("/api/v1/rag")
public class RagController {
    
    @Autowired
    private RagService ragService;
    
    @Autowired
    private pangea.hiagent.web.service.AgentService agentService;
    
    /**
     * 文档检索接口
     */
    @PostMapping("/search")
    public ApiResponse<List<org.springframework.ai.document.Document>> searchDocuments(
            @RequestBody SearchRequest request) {
        try {
            log.info("开始文档检索，查询: {}", request.getQuery());
            
            List<org.springframework.ai.document.Document> results = ragService.searchDocuments(
                request.getQuery(),
                request.getCollectionIds(),
                request.getTopK(),
                request.getScoreThreshold()
            );
            
            log.info("文档检索完成，返回 {} 个结果", results != null ? results.size() : 0);
            return ApiResponse.success(results, "文档检索成功");
        } catch (Exception e) {
            log.error("文档检索失败", e);
            return ApiResponse.error(4002, "文档检索失败: " + e.getMessage());
        }
    }
    
    /**
     * 获取所有文档列表接口（供Dashboard使用）
     */
    @GetMapping("/documents")
    public ApiResponse<List<Object>> getAllDocuments() {
        try {
            log.info("获取所有文档列表");
            
            // 由于VectorStore可能未配置，我们返回空列表而不是抛出异常
            List<Object> documents = List.of();
            
            log.info("获取文档列表完成，返回 {} 个文档", documents.size());
            return ApiResponse.success(documents, "获取文档列表成功");
        } catch (Exception e) {
            log.error("获取文档列表失败", e);
            return ApiResponse.error(4002, "获取文档列表失败: " + e.getMessage());
        }
    }
    
    /**
     * RAG增强问答接口
     */
    @PostMapping("/qa")
    public ApiResponse<String> ragQa(@RequestBody RagQaRequest request) {
        try {
            log.info("开始RAG增强问答，Agent ID: {}, 查询: {}", request.getAgentId(), request.getQuery());
            
            // 根据Agent ID获取Agent对象
            Agent agent = null;
            if (request.getAgentId() != null && !request.getAgentId().isEmpty()) {
                agent = agentService.getAgent(request.getAgentId());
                if (agent == null) {
                    log.warn("未找到指定的Agent: {}", request.getAgentId());
                    return ApiResponse.error(4003, "未找到指定的Agent: " + request.getAgentId());
                }
                log.debug("成功获取Agent: {} ({}), 启用RAG: {}", 
                         agent.getName(), agent.getId(), agent.getEnableRag());
            } else {
                log.warn("Agent ID为空，无法执行RAG增强问答");
                return ApiResponse.error(4004, "Agent ID不能为空");
            }
            
            // 调用RagService的实际方法
            String result = ragService.ragQa(agent, request.getQuery());
            
            if (result == null) {
                log.warn("RAG增强问答返回结果为空");
                return ApiResponse.error(4005, "RAG增强问答未返回有效结果");
            }
            
            log.info("RAG增强问答完成");
            return ApiResponse.success(result, "RAG增强问答成功");
        } catch (Exception e) {
            log.error("RAG增强问答失败", e);
            return ApiResponse.error(4002, "RAG增强问答失败: " + e.getMessage());
        }
    }
    
    /**
     * 获取检索统计信息
     */
    @GetMapping("/stats")
    public ApiResponse<Map<String, Object>> getRetrievalStats(
            @RequestParam(required = false) List<String> collectionIds) {
        try {
            log.info("获取检索统计信息");
            
            Map<String, Object> stats = ragService.getRetrievalStats(collectionIds);
            
            log.info("获取检索统计信息完成");
            return ApiResponse.success(stats, "获取统计信息成功");
        } catch (Exception e) {
            log.error("获取检索统计信息失败", e);
            return ApiResponse.error(4002, "获取统计信息失败: " + e.getMessage());
        }
    }
    
    /**
     * 文档检索请求类
     */
    @Data
    public static class SearchRequest {
        private String query;
        private List<String> collectionIds;
        private Integer topK;
        private Double scoreThreshold;
    }
    
    /**
     * RAG问答请求类
     */
    @Data
    public static class RagQaRequest {
        private String agentId;
        private String query;
    }
}