package pangea.hiagent.agent.service;

import lombok.extern.slf4j.Slf4j;
import pangea.hiagent.agent.data.ErrorEventDataBuilder;
import pangea.hiagent.agent.data.MapPoolService;
import pangea.hiagent.agent.data.TokenEventDataBuilder;
import pangea.hiagent.agent.data.WorkPanelEvent;

import org.springframework.stereotype.Service;
import org.springframework.web.servlet.mvc.method.annotation.SseEmitter;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.Map;
import java.util.concurrent.*;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.io.IOException;
import java.util.concurrent.ScheduledFuture;

/**
 * 统一SSE服务类
 * 整合SSE连接管理、生命周期管理、回调处理等功能
 */
@Slf4j
@Service
public class UserSseService {

    // 存储所有活动的 emitter
    private final List<SseEmitter> emitters = new CopyOnWriteArrayList<>();

    // 存储用户ID到SSE Emitter的映射关系
    private final ConcurrentMap<String, SseEmitter> userEmitters = new ConcurrentHashMap<>();

    // 存储SSE Emitter到用户ID的反向映射关系（用于快速查找）
    private final ConcurrentMap<SseEmitter, String> emitterUsers = new ConcurrentHashMap<>();

    // 存储已完成的emitter集合，用于快速检查状态
    private final Set<SseEmitter> completedEmitters = ConcurrentHashMap.newKeySet();

    // 心跳任务执行器 - 使用共享线程池以提高资源利用率
    private final ScheduledExecutorService heartbeatExecutor;

    private final TokenEventDataBuilder tokenEventDataBuilder;

    private final ErrorEventDataBuilder errorEventDataBuilder;

    private final MapPoolService mapPoolService;

    // SSE超时时间（毫秒）
    private static final long SSE_TIMEOUT = 0L; // 0表示不使用默认超时，由心跳机制管理连接

    public UserSseService(TokenEventDataBuilder tokenEventDataBuilder, ErrorEventDataBuilder errorEventDataBuilder, MapPoolService mapPoolService) {
        this.tokenEventDataBuilder = tokenEventDataBuilder;
        this.errorEventDataBuilder = errorEventDataBuilder;
        this.mapPoolService = mapPoolService;
        this.heartbeatExecutor = Executors.newScheduledThreadPool(2);
    }

    /**
     * 创建并注册SSE连接
     * 
     * @param userId 用户ID
     * @return SSE Emitter
     */
    public SseEmitter createAndRegisterConnection(String userId) {
        log.debug("开始为用户 {} 创建SSE连接", userId);

        // 创建 SSE emitter
        SseEmitter emitter = createEmitter();
        log.debug("SSE Emitter创建成功");

        // 注册用户的SSE连接
        registerSession(userId, emitter);
        log.debug("用户 {} 的SSE连接注册成功", userId);

        // 注册 emitter 回调
        registerCallbacks(emitter, userId);
        log.debug("SSE Emitter回调注册成功");

        // 启动心跳机制
        startHeartbeat(emitter, new AtomicBoolean(false));
        log.debug("心跳机制启动成功");

        log.info("用户 {} 的SSE连接创建和注册完成", userId);
        return emitter;
    }

    /**
     * 创建SSE发射器
     * 
     * @return SSE发射器
     */
    public SseEmitter createEmitter() {
        SseEmitter emitter = new SseEmitter(SSE_TIMEOUT);
        emitters.add(emitter);
        // 启动心跳机制，确保新创建的连接有心跳
        startHeartbeat(emitter, new AtomicBoolean(false));
        return emitter;
    }

    /**
     * 注册用户的SSE连接
     * 如果该用户已有连接，则先关闭旧连接再注册新连接
     * 
     * @param userId  用户ID
     * @param emitter SSE Emitter
     * @return true表示注册成功，false表示注册失败
     */
    public boolean registerSession(String userId, SseEmitter emitter) {
        if (userId == null || userId.isEmpty() || emitter == null) {
            log.warn("注册SSE会话失败：用户ID或Emitter为空");
            return false;
        }

        try {
            // 检查该用户是否已有连接
            SseEmitter existingEmitter = userEmitters.get(userId);
            if (existingEmitter != null) {
                log.info("用户 {} 已有SSE连接，关闭旧连接", userId);
                try {
                    existingEmitter.complete();
                } catch (Exception e) {
                    log.warn("关闭用户 {} 的旧SSE连接时发生异常: {}", userId, e.getMessage());
                }
                // 从映射表中移除旧连接
                userEmitters.remove(userId);
                emitterUsers.remove(existingEmitter);
            }

            // 注册新连接
            userEmitters.put(userId, emitter);
            emitterUsers.put(emitter, userId);

            log.info("用户 {} 的SSE连接注册成功", userId);
            return true;
        } catch (Exception e) {
            log.error("注册用户 {} 的SSE连接时发生异常", userId, e);
            return false;
        }
    }

    /**
     * 获取用户的SSE连接
     * 
     * @param userId 用户ID
     * @return SSE Emitter，如果不存在则返回null
     */
    public SseEmitter getSession(String userId) {
        return userEmitters.get(userId);
    }

    /**
     * 通用连接关闭处理方法
     * 
     * @param emitter        SSE Emitter
     * @param connectionType 连接类型（用于日志）
     */
    private void handleConnectionClose(SseEmitter emitter, String connectionType) {
        if (emitter == null) {
            return;
        }

        try {
            // 添加到已完成集合
            completedEmitters.add(emitter);
            
            // 检查emitter是否已经完成，避免重复关闭
            if (!isEmitterCompleted(emitter)) {
                try {
                    log.info("handleConnectionClose: SSE连接已关闭");
                    emitter.complete();
                } catch (Exception e) {
                    log.debug("完成emitter时发生异常（可能是由于已关闭）: {}", e.getMessage());
                }
            }

            // 从映射表中移除连接
            String userId = emitterUsers.remove(emitter);
            if (userId != null) {
                userEmitters.remove(userId);
            }
            emitters.remove(emitter);

            log.debug("SSE连接{}，用户: {}", connectionType, userId);
        } catch (Exception e) {
            log.error("处理SSE连接{}事件时发生异常", connectionType, e);
        }
    }

    /**
     * 处理连接完成事件
     * 
     * @param emitter SSE Emitter
     */
    public void handleConnectionCompletion(SseEmitter emitter) {
        handleConnectionClose(emitter, "完成");
    }

    /**
     * 处理连接超时事件
     * 
     * @param emitter SSE Emitter
     */
    public void handleConnectionTimeout(SseEmitter emitter) {
        handleConnectionClose(emitter, "超时");
    }

    /**
     * 处理连接错误事件
     * 
     * @param emitter SSE Emitter
     */
    public void handleConnectionError(SseEmitter emitter) {
        handleConnectionClose(emitter, "错误");
    }

    /**
     * 移除SSE发射器
     * 
     * @param emitter SSE发射器
     */
    public void removeEmitter(SseEmitter emitter) {
        if (emitter != null) {
            completedEmitters.add(emitter); // 添加到已完成集合
            if (emitters.remove(emitter)) {
                log.debug("已移除SSE Emitter，剩余连接数: {}", emitters.size());
            }
        }
    }

    /**
     * 启动心跳机制
     * 
     * @param emitter     SSE发射器
     * @param isCompleted 是否已完成
     */
    public void startHeartbeat(SseEmitter emitter, AtomicBoolean isCompleted) {
        if (emitter == null) {
            log.warn("SSE发射器为空，无法启动心跳机制");
            return;
        }

        // 用于追踪心跳失败次数
        AtomicInteger consecutiveFailures = new AtomicInteger(0);

        // 使用数组包装ScheduledFuture以解决Lambda中的变量访问问题
        final ScheduledFuture<?>[] heartbeatTaskRef = new ScheduledFuture<?>[1];

        // 创建心跳任务并保存ScheduledFuture引用
        heartbeatTaskRef[0] = heartbeatExecutor.scheduleAtFixedRate(() -> {
            try {
                // 检查emitter是否已经完成
                if (isCompleted.get() || isEmitterCompleted(emitter)) {
                    log.debug("SSE Emitter已完成或无效，取消心跳任务");
                    if (heartbeatTaskRef[0] != null && !heartbeatTaskRef[0].isCancelled()) {
                        heartbeatTaskRef[0].cancel(true);
                    }
                    return;
                }

                // 发送心跳事件
                boolean heartbeatSuccess = sendHeartbeat(emitter);

                if (heartbeatSuccess) {
                    // 如果心跳成功，重置失败计数
                    consecutiveFailures.set(0);

                    log.debug("心跳发送成功，连续失败次数重置为0");

                    // 心跳成功后，连接保持活动状态，不需要额外操作，因为SSE_TIMEOUT为0
                } else {
                    // 心跳失败，增加失败计数
                    int currentFailures = consecutiveFailures.incrementAndGet();
                    log.debug("心跳连续失败次数: {}", currentFailures);

                    // 如果心跳连续失败达到阈值，启动延迟关闭
                    if (currentFailures >= 2) { // 连续2次失败后，启动30秒延迟关闭
                        log.warn("心跳连续失败{}次，启动30秒延迟关闭机制", currentFailures);

                        // 调度一个延迟任务来关闭连接
                        heartbeatExecutor.schedule(() -> {
                            if (!isCompleted.get() && !isEmitterCompleted(emitter)) {
                                log.info("30秒延迟到期，主动关闭SSE连接");

                                // 首先取消心跳任务
                                if (heartbeatTaskRef[0] != null && !heartbeatTaskRef[0].isCancelled()) {
                                    heartbeatTaskRef[0].cancel(true);
                                    log.debug("心跳任务已取消");
                                }

                                // 然后关闭SSE连接
                                try {
                                    if (!isEmitterCompleted(emitter)) {
                                        emitter.complete();
                                        log.info("startHeartbeat: SSE连接已关闭");
                                    }
                                } catch (Exception ex) {
                                    log.debug("关闭SSE连接时发生异常（可能是由于已关闭）: {}", ex.getMessage());
                                }
                            } else {
                                log.debug("SSE连接已完成或已关闭，跳过延迟关闭");
                            }
                        }, 30, TimeUnit.SECONDS);

                        // 立即取消当前心跳任务
                        if (heartbeatTaskRef[0] != null && !heartbeatTaskRef[0].isCancelled()) {
                            heartbeatTaskRef[0].cancel(true);
                            log.debug("心跳任务因连续失败而被取消");
                        }
                    }
                }

            } catch (Exception e) {
                log.error("心跳任务执行异常: {}", e.getMessage(), e);
            }
        }, 20, 20, TimeUnit.SECONDS); // 每20秒发送一次心跳，确保前端60秒超时前至少收到2次心跳
    }

    /**
     * 注册回调函数
     * 
     * @param emitter SSE发射器
     * @param userId  用户ID（可选，用于完整的连接管理）
     */
    public void registerCallbacks(SseEmitter emitter, String... userId) {
        boolean hasUserId = userId != null && userId.length > 0 && userId[0] != null;

        emitter.onCompletion(() -> {
            log.debug("【注册回调函数】SSE连接完成");
            if (hasUserId) {
                handleConnectionCompletion(emitter);
            } else {
                removeEmitter(emitter);
            }
        });

        emitter.onError((Throwable t) -> {
            String errorMessage = t.getMessage();
            String errorType = t.getClass().getSimpleName();
            log.debug("SSE连接发生错误 - 类型: {}, 消息: {}", errorType, errorMessage);
            if (hasUserId) {
                handleConnectionError(emitter);
            } else {
                removeEmitter(emitter);
            }
        });

        emitter.onTimeout(() -> {
            log.warn("SSE连接超时");
            try {
                // 检查emitter是否已经完成，避免重复关闭
                if (!isEmitterCompleted(emitter)) {
                    log.info("registerCallbacks: SSE连接已关闭");
                    emitter.complete();
                }
            } catch (Exception e) {
                log.debug("关闭SSE连接时发生异常（可能是由于已关闭）: {}", e.getMessage());
            }
            if (hasUserId) {
                handleConnectionTimeout(emitter);
            } else {
                removeEmitter(emitter);
            }
        });
    }

    /**
     * 完成SSE发射器
     * 
     * @param emitter     SSE发射器
     * @param isCompleted 是否已完成
     */
    public void completeEmitter(SseEmitter emitter, AtomicBoolean isCompleted) {
        if (isCompleted != null) {
            isCompleted.set(true);
        }
        if (emitter != null) {
            try {
                // 检查emitter是否已经完成，避免重复关闭
                if (isEmitterCompleted(emitter)) {
                    log.debug("Emitter已经完成，跳过关闭操作");
                    return;
                }
                log.info("completeEmitter: Emitter已成功关闭");
                emitter.complete();
                completedEmitters.add(emitter); // 添加到已完成集合
                log.debug("Emitter已成功关闭");
            } catch (IllegalStateException e) {
                log.debug("Emitter已经关闭: {}", e.getMessage());
                completedEmitters.add(emitter); // 添加到已完成集合
            } catch (Exception e) {
                log.warn("完成Emitter时发生异常: {}", e.getMessage());
                completedEmitters.add(emitter); // 添加到已完成集合
            }
        }
    }

    /**
     * 检查SSE Emitter是否仍然有效
     * 职责：提供轻量级的连接有效性检查
     * 
     * @param emitter 要检查的emitter
     * @return 如果有效返回true，否则返回false
     */
    public boolean isEmitterValid(SseEmitter emitter) {
        if (emitter == null) {
            return false;
        }

        // 首先检查是否已经完成，避免不必要的事件发送
        if (isEmitterCompleted(emitter)) {
            return false;
        }

        // 检查逻辑，仅通过尝试发送ping事件来验证连接状态
        try {
            // 尝试发送一个空事件来检查连接状态
            // 注意：这个方法会实际发送一个事件到客户端，这可能不是理想的方式
            emitter.send(SseEmitter.event().name("ping").data("").build());
            return true;
        } catch (Exception ex) {
            // 如果出现任何异常，认为连接已失效
            return false;
        }
    }

    /**
     * 安全检查SSE Emitter是否仍然有效（不发送实际事件）
     * 职责：提供非侵入性的连接有效性检查
     * 
     * @param emitter 要检查的emitter
     * @return 如果有效返回true，否则返回false
     */
    public boolean isEmitterValidSafe(SseEmitter emitter) {
        if (emitter == null) {
            return false;
        }

        // 检查是否已经完成，而不发送任何事件
        return !isEmitterCompleted(emitter);
    }

    /**
     * 检查SSE Emitter是否已经完成
     * 使用高效的方式检查完成状态，优先检查本地集合
     * 
     * @param emitter 要检查的emitter
     * @return 如果已完成返回true，否则返回false
     */
    public boolean isEmitterCompleted(SseEmitter emitter) {
        if (emitter == null) {
            return true; // 认为null emitter是已完成的
        }

        // 首先检查本地集合，避免网络操作
        if (completedEmitters.contains(emitter)) {
            return true;
        }

        // 检查emitter是否在活动列表中
        if (!emitters.contains(emitter)) {
            completedEmitters.add(emitter); // 添加到已完成集合
            return true;
        }

        // 尝试使用安全的方式检查emitter状态，避免发送实际事件
        // 注意：这个方法不会捕获所有连接问题，但能有效避免UT005023异常
        return false;
    }

    /**
     * 安全发送SSE事件，处理所有异常情况
     * 
     * @param emitter   SSE发射器
     * @param eventName 事件名称
     * @param data      事件数据
     * @return 是否发送成功
     */
    private boolean safeSendEvent(SseEmitter emitter, String eventName, Object data) {
        // 参数验证
        if (emitter == null || eventName == null || eventName.isEmpty() || data == null) {
            log.warn("参数验证失败，无法发送事件");
            return false;
        }

        // 检查emitter是否已经完成
        if (isEmitterCompleted(emitter)) {
            log.debug("SSE emitter已完成，跳过发送{}事件", eventName);
            return false;
        }

        try {
            emitter.send(SseEmitter.event().name(eventName).data(data));
            return true;
        } catch (IllegalStateException e) {
            // 处理 emitter 已关闭的情况
            log.debug("无法发送{}事件，emitter已关闭: {}", eventName, e.getMessage());
            return false;
        } catch (Exception e) {
            log.error("发送{}事件失败: {}", eventName, e.getMessage(), e);
            return false;
        }
    }

    /**
     * 发送SSE事件
     * 职责：统一发送SSE事件的基础方法
     * 
     * @param emitter   SSE发射器
     * @param eventName 事件名称
     * @param data      事件数据
     * @throws IOException IO异常
     */
    public void sendEvent(SseEmitter emitter, String eventName, Object data) throws IOException {
        safeSendEvent(emitter, eventName, data);
    }

    /**
     * 发送心跳事件
     * 
     * @param emitter SSE发射器
     * @return 是否成功发送心跳
     */
    public boolean sendHeartbeat(SseEmitter emitter) {
        if (emitter == null) {
            log.warn("SSE发射器为空，无法发送心跳事件");
            return false;
        }

        // 发送心跳事件
        long heartbeatTimestamp = System.currentTimeMillis();
        boolean success = safeSendEvent(emitter, "heartbeat", heartbeatTimestamp);

        if (success) {
            log.debug("[心跳] 成功发送心跳事件，时间戳: {}", heartbeatTimestamp);
        }

        return success;
    }

    /**
     * 发送工作面板事件给指定的SSE连接
     * 
     * @param emitter SSE发射器
     * @param event   工作面板事件
     * @throws IOException IO异常
     */
    public void sendWorkPanelEvent(WorkPanelEvent event) throws IOException {
        if (event == null) {
            log.warn("工作面板事件为空，无法发送事件");
            return;
        }

        try {
            // 构建事件数据
            Map<String, Object> data = buildWorkPanelEventData(event);

            if (data != null) {
                log.debug("准备发送工作面板事件: 类型={}, 事件内容={}", event.getType(), event);
                log.debug("事件数据: {}", data);

                SseEmitter emitter = getSession(event.getUserId());

                if (emitter == null) {
                    log.debug("未找到用户 {} 的SSE连接，跳过发送事件", event.getUserId());
                    return;
                }

                // 发送事件
                emitter.send(SseEmitter.event().name("message").data(data));

                log.debug("工作面板事件发送成功: 类型={}", event.getType());
            } else {
                log.warn("构建事件数据失败，无法发送事件: 类型={}", event.getType());
            }
        } catch (IllegalStateException e) {
            // 处理 emitter 已关闭的情况
            log.debug("无法发送工作面板事件，emitter已关闭或完成: {}", e.getMessage());
            // 不重新抛出异常，避免影响主流程
        } catch (Exception e) {
            // 记录详细错误信息，但不中断主流程
            log.error("发送工作面板事件失败: 类型={}, 错误={}", event.getType(), e.getMessage(), e);

            // 其他异常不重新抛出，避免影响主流程
        }
    }

    /**
     * 构建工作面板事件数据
     * 
     * @param event 工作面板事件
     * @return 事件数据
     */
    private Map<String, Object> buildWorkPanelEventData(WorkPanelEvent event) {
        if (event == null) {
            return null;
        }
        
        // 从对象池获取Map，避免频繁创建对象
        Map<String, Object> data = mapPoolService.acquireMap();
        
        // 设置基础属性
        data.put("type", event.getType());
        data.put("timestamp", event.getTimestamp());
        data.put("title", event.getTitle());
        data.put("content", event.getContent());
        // data.put("userId", event.getUserId());

        if(event.getMetadata() != null) {
            data.putAll(event.getMetadata());
        }
        
        return data;
    }

    /**
     * 发送连接成功事件
     * 
     * @param emitter SSE发射器
     * @throws IOException IO异常
     */
    public void sendConnectedEvent(SseEmitter emitter) throws IOException {
        if (emitter == null) {
            log.warn("SSE发射器为空，无法发送连接成功事件");
            return;
        }

        WorkPanelEvent connectedEvent = WorkPanelEvent.builder()
                .type("observation")
                .title("连接成功")
                .timestamp(System.currentTimeMillis())
                .build();

        Map<String, Object> data = buildWorkPanelEventData(connectedEvent);
        boolean success = safeSendEvent(emitter, "message", data);

        if (success) {
            log.debug("已发送连接成功事件");
        }
    }

    /**
     * 发送Token事件
     * 
     * @param emitter SSE发射器
     * @param token   Token内容
     * @throws IOException IO异常
     */
    public void sendTokenEvent(SseEmitter emitter, String token) throws IOException {
        if (emitter == null || token == null) {
            log.warn("SSE发射器或Token为空，无法发送Token事件");
            return;
        }

        try {
            // 构建token事件数据
            Map<String, Object> data = tokenEventDataBuilder.createOptimizedTokenEventData(token);

            if (data != null) {
                // 发送事件
                safeSendEvent(emitter, "token", data);
            } else {
                log.warn("构建token事件数据失败，无法发送事件");
            }
        } catch (Exception e) {
            log.error("发送token事件失败: token长度={}, 错误={}", token.length(), e.getMessage(), e);
        }
    }

    /**
     * 发送错误事件
     * 
     * @param emitter      SSE发射器
     * @param errorMessage 错误信息
     * @throws IOException IO异常
     */
    public void sendErrorEvent(SseEmitter emitter, String errorMessage) throws IOException {
        if (emitter == null || errorMessage == null) {
            log.warn("SSE发射器或错误信息为空，无法发送错误事件");
            return;
        }

        try {
            // 构建错误事件数据
            Map<String, Object> data = errorEventDataBuilder.createErrorEventData(errorMessage);

            if (data != null) {
                // 发送事件
                safeSendEvent(emitter, "error", data);
            } else {
                log.warn("构建错误事件数据失败，无法发送事件");
            }
        } catch (Exception e) {
            log.error("发送错误事件失败: 错误信息={}, 错误={}", errorMessage, e.getMessage(), e);
        }
    }

    /**
     * 获取所有活动的emitters
     * 
     * @return 所有活动的emitters列表
     */
    public List<SseEmitter> getEmitters() {
        return new ArrayList<>(emitters);
    }

    /**
     * 关闭SSE发射器
     * 
     * @param emitter SSE发射器
     */
    public void closeEmitter(SseEmitter emitter) {
        if (emitter == null) {
            return;
        }
        
        try {
            if (!isEmitterCompleted(emitter)) {
                log.info("closeEmitter: Emitter已成功关闭");
                emitter.complete();
            }
            // 从活动连接列表中移除
            removeEmitter(emitter);
        } catch (Exception e) {
            log.warn("关闭SSE发射器时发生异常: {}", e.getMessage());
        }
    }

    /**
     * 销毁资源
     */
    public void destroy() {
        // 关闭线程池
        heartbeatExecutor.shutdown();
        try {
            if (!heartbeatExecutor.awaitTermination(5, TimeUnit.SECONDS)) {
                heartbeatExecutor.shutdownNow();
            }
        } catch (InterruptedException e) {
            heartbeatExecutor.shutdownNow();
            Thread.currentThread().interrupt();
        }
    }
}