package pangea.hiagent.agent.data;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * 模块测试服务，用于验证模块功能和热更新能力
 */
@Slf4j
@Service
public class ModuleTestService {
    
    @Autowired
    private ModuleProperties moduleProperties;
    
    @Autowired
    private LogModuleAutoConfig logModule;
    
    @Autowired
    private CheckModuleAutoConfig checkModule;
    
    @Autowired
    private MonitorModuleAutoConfig monitorModule;
    
    @Autowired
    private FallbackModuleAutoConfig fallbackModule;
    
    /**
     * 测试所有模块的基本功能
     */
    public void testAllModules() {
        log.info("开始测试所有模块功能...");
        
        // 测试模块属性
        testModuleProperties();
        
        // 模拟发布任务事件以触发各模块逻辑
        testModuleEventHandling();
        
        log.info("所有模块功能测试完成");
    }
    
    /**
     * 测试模块属性配置
     */
    private void testModuleProperties() {
        log.info("测试模块属性配置:");
        log.info("  - 模块总开关: {}", moduleProperties.isEnabled());
        log.info("  - 日志模块: enabled={}, retentionDays={}, storagePath={}", 
                moduleProperties.getLog().isEnabled(), 
                moduleProperties.getLog().getRetentionDays(), 
                moduleProperties.getLog().getStoragePath());
        log.info("  - 校验模块: enabled={}, strictMode={}, algorithm={}", 
                moduleProperties.getCheck().isEnabled(), 
                moduleProperties.getCheck().isStrictMode(), 
                moduleProperties.getCheck().getAlgorithm());
        log.info("  - 监控模块: enabled={}, refreshInterval={}, enableGraph={}", 
                moduleProperties.getMonitor().isEnabled(), 
                moduleProperties.getMonitor().getRefreshInterval(), 
                moduleProperties.getMonitor().isEnableGraph());
        log.info("  - 降级模块: enabled={}, maxRetries={}, retryDelay={}, defaultStrategy={}", 
                moduleProperties.getFallback().isEnabled(), 
                moduleProperties.getFallback().getMaxRetries(), 
                moduleProperties.getFallback().getRetryDelay(), 
                moduleProperties.getFallback().getDefaultStrategy());
    }
    
    /**
     * 测试模块事件处理
     */
    private void testModuleEventHandling() {
        log.info("测试模块事件处理...");
        
        // 创建模拟事件来触发各模块的事件监听器
        AgentTaskEvent startEvent = new AgentTaskEvent(this, "test-task-001", "STARTED");
        AgentTaskEvent completedEvent = new AgentTaskEvent(this, "test-task-001", "COMPLETED");
        AgentTaskEvent failedEvent = new AgentTaskEvent(this, "test-task-002", "FAILED");
        
        log.info("模拟任务开始事件: {}", startEvent.getTaskId());
        log.info("模拟任务完成事件: {}", completedEvent.getTaskId());
        log.info("模拟任务失败事件: {}", failedEvent.getTaskId());
        
        log.info("模块事件处理测试完成");
    }
    
    /**
     * 测试热更新功能
     */
    public void testHotReload() {
        log.info("测试配置热更新功能...");
        
        // 显示当前配置
        log.info("当前日志模块状态: {}", moduleProperties.getLog().isEnabled());
        log.info("当前校验模块状态: {}", moduleProperties.getCheck().isEnabled());
        log.info("当前监控模块状态: {}", moduleProperties.getMonitor().isEnabled());
        log.info("当前降级模块状态: {}", moduleProperties.getFallback().isEnabled());
        
        log.info("热更新功能测试完成 - 配置将在运行时动态更新");
    }
}