package pangea.hiagent.tool;

import lombok.extern.slf4j.Slf4j;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NoArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.aop.support.AopUtils;
import org.springframework.context.annotation.Lazy;
import pangea.hiagent.model.Tool;
import pangea.hiagent.web.repository.ToolRepository;

import java.util.*;
import java.lang.reflect.Method;

/**
 * 工具Bean名称初始化器 - 反向扫描版
 * 
 * 核心理念: 从Spring容器反向扫描所有工具Bean，与数据库tool表进行同步
 * 
 * 工作流程:
 * 1. 扫描Spring容器中所有Bean
 * 2. 识别工具类（通过检查类中是否有方法带有@Tool注解）
 * 3. 与数据库tool表进行对比：
 *    - 数据库已存在 → 更新beanName配置
 *    - 数据库不存在 → 可选创建tool记录（通过配置控制）
 *    - 数据库有但Bean不存在 → 记录警告，留给管理员处理
 * 4. 确保所有Bean都有对应的tool记录且beanName配置正确
 * 
 * 优势:
 * - 主动发现系统中的所有工具Bean
 * - 自动保持数据库与代码的同步
 * - 减少手动配置的工作量
 * - 支持热部署新增的工具
 */
@Slf4j
@Component
@Lazy
public class ToolBeanNameInitializer {

    @Autowired
    private ToolRepository toolRepository;
    
    @Autowired
    private ApplicationContext applicationContext;
    

    
    /**
     * 手动触发：扫描Spring容器中的工具Bean并与数据库同步
     * 
     * 该方法不再在应用启动时自动执行，而是通过管理界面手动触发
     */
    @Transactional(rollbackFor = Exception.class)
    public void initializeToolBeanNamesManually() {
        try {
            log.info("========== [工具Bean初始化] 开始扫描工具Bean ==========");
            
            // 第一步：从Spring容器扫描所有工具Bean
            Map<String, ToolBeanInfo> discoveredTools = scanToolBeansFromSpring();
            log.info("[工具Bean初始化] 从Spring容器发现了{}个工具Bean", discoveredTools.size());
            
            if (discoveredTools.isEmpty()) {
                log.info("[工具Bean初始化] Spring容器中没有发现工具Bean，跳过初始化");
                return;
            }
            
            // 第二步：从数据库加载所有工具记录
            Map<String, Tool> databaseTools = loadToolsFromDatabase();
            log.info("[工具Bean初始化] 从数据库加载了{}个工具记录", databaseTools.size());
            
            // 第三步：对比并同步
            synchronizeToolsWithDatabase(discoveredTools, databaseTools);
            
            log.info("========== [工具Bean初始化] 扫描和同步完成 ==========");
            
        } catch (Exception e) {
            log.error("[工具Bean初始化] 初始化工具Bean名称映射时发生异常", e);
        }
    }
    
    /**
     * 从Spring容器中扫描所有工具Bean
     * 
     * 识别规则:
     * 1. 类名包含"Tool"关键字
     * 2. 被@Component或@Service标注
     * 3. 不是Spring框架自身的bean
     * 
     * @return 工具Bean信息映射 (beanName -> ToolBeanInfo)
     */
    private Map<String, ToolBeanInfo> scanToolBeansFromSpring() {
        Map<String, ToolBeanInfo> toolBeans = new HashMap<>();
            
        String[] beanNames = null;
        int toolClassCount = 0;
        int processedCount = 0;
            
        try {
            beanNames = applicationContext.getBeanDefinitionNames();
            log.info("[工具Bean扫描] 正在扫描 Spring 容器中的 {} 个Bean", beanNames.length);
            
            for (String beanName : beanNames) {
                try {
                    Object bean = null;
                    try {
                        bean = applicationContext.getBean(beanName);
                    } catch (Exception e) {
                        // Bean实例化失败，但继续检查Bean定义
                        log.debug("[工具Bean扫描] Bean实例化失败({}): {}", beanName, e.getClass().getSimpleName());
                        continue;
                    }
                    
                    if (bean == null) {
                        continue;
                    }
                    
                    Class<?> beanClass = bean.getClass();
                    String simpleClassName = beanClass.getSimpleName();
                    String packageName = beanClass.getPackage() != null ? beanClass.getPackage().getName() : "";
                    
                    // 检查是否为工具类（检查类中是否有@Tool注解的方法）
                    if (isToolClass(beanClass, packageName)) {
                        ToolBeanInfo info = new ToolBeanInfo();
                        info.setBeanName(beanName);
                        info.setSimpleClassName(simpleClassName);
                        info.setFullClassName(beanClass.getName());
                        info.setPackageName(packageName);
                        info.setBeanClass(beanClass);
                        info.setInstance(bean);
                        
                        // 推导工具名称：获取带@Tool注解的方法名
                        String toolName = deriveToolName(beanClass);
                        info.setDerivedToolName(toolName);
                        
                        toolBeans.put(beanName, info);
                        log.debug("[工具Bean扫描] 发现工具Bean: {} (class: {}, toolName: {})", 
                            beanName, simpleClassName, toolName);
                        toolClassCount++;
                    }
                    processedCount++;
                } catch (Exception e) {
                    log.error("[工具Bean扫描] 处理Bean'{}' 时发生异常: {} | 错误信息: {} | 堆栈信息:", beanName, e.getClass().getSimpleName(), e.getMessage(), e);
                }
            }
        } catch (Exception e) {
            log.error("[工具Bean扫描] 扫描Spring容器时发生异常: {} | 错误信息: {} | 堆栈信息:", e.getClass().getSimpleName(), e.getMessage(), e);
        }
        
        log.info("[工具Bean扫描] 扫描完成：总计扫描{}Bean，发现{}Tool类，实际获得{}Tool", beanNames.length, toolClassCount, toolBeans.size());
        return toolBeans;
    }
    
/**
     * 判断指定的Bean Class（可能是代理类）是否为指定包下、带有@Tool注解方法的工具类
     *
     * @param beanClass   Bean的Class对象（可能是代理类）
     * @param packageName 包名（例如：com.example.demo），会匹配该包及其子包
     * @return true：是符合条件的工具类；false：不符合
     */
    public boolean isToolClass(Class<?> beanClass, String packageName) {
        // 防御性判断：参数为空时直接返回false
        if (beanClass == null || packageName == null || packageName.isBlank()) {
            return false;
        }

             // 1. 排除Spring框架自身的bean
        if (packageName.startsWith("org.springframework") || packageName.startsWith("java.")
            || packageName.startsWith("javax.") || packageName.startsWith("org.aspectj") 
            || packageName.startsWith("jakarta") || packageName.startsWith("org.jakarta")
            || packageName.startsWith("com.baomidou") || packageName.startsWith("org.apache")) {
            return false;
        }

        // 步骤2：获取原始目标类（穿透Spring AOP代理类）
        Class<?> targetClass = AopUtils.getTargetClass(beanClass);
        // 若传入的不是代理类，AopUtils.getTargetClass会直接返回原类，不影响逻辑

        // 步骤3：判断原始类是否属于指定包及其子包
        String className = targetClass.getName();
        if (!className.startsWith(packageName + ".")) {
            return false;
        }

        // 步骤3：检查原始类的方法是否带有@Tool注解（包括自身声明和继承的public方法）
        // 获取类的所有public方法（包括父类的public方法）
        Method[] methods = targetClass.getMethods();
        for (Method method : methods) {
            // 检查方法是否直接带有@Tool注解
            if (method.isAnnotationPresent(org.springframework.ai.tool.annotation.Tool.class)) {
                return true;
            }
        }

        return false;
    }

        
    /**
     * 从Bean类推导工具名称
     * 
     * 规范：必须使用类名作为工具名，禁止使用方法名
     * 支持代理类的处理
     * 
     * 推导规则:
     * 1. 获取实际目标类的简单名称（不含包名、不含代理标记）
     * 2. 如果以"Tool"结尾，则去掉"Tool"后缀
     * 3. 转换为小驼峰格式（首字母小写）
     * 
     * 例如:
     * - 类名 "CalculatorTool" → 工具名 "calculator"
     * - 类名 "StorageFileAccessTool" → 工具名 "storageFileAccess"
     * - 类名 "CodeAnalyzer" → 工具名 "codeAnalyzer"
     * - 代理类 "CalculatorTool$$EnhancerBySpringCGLIB$$..." → 工具名 "calculator"
     * 
     * @param beanClass Bean的Class对象（可能是代理类）
     * @return 推导的工具名称
     */
    private String deriveToolName(Class<?> beanClass) {
        // 首先获取实际的目标类（处理代理）
        Class<?> targetClass = AopUtils.getTargetClass(beanClass);
        String simpleClassName = targetClass.getSimpleName();
        
        // 从类名推导工具名
        String toolName = simpleClassName;
        
        // 如果类名以"Tool"结尾，则去掉该后缀
        if (toolName.endsWith("Tool")) {
            toolName = toolName.substring(0, toolName.length() - 4);
        }
        
        // 如果推导结果为空，则返回"tool"
        if (toolName.isEmpty()) {
            return "tool";
        }
        
        // 转换为小驼峰格式：首字母小写
        return toolName.substring(0, 1).toLowerCase() + toolName.substring(1);
    }
    
    /**
     * 从数据库加载所有未被删除的工具记录
     * 
     * @return 工具记录映射 (toolName -> Tool)
     */
    private Map<String, Tool> loadToolsFromDatabase() {
        Map<String, Tool> tools = new HashMap<>();
        
        try {
            List<Tool> allTools = toolRepository.selectList(null);
            if (allTools == null) {
                return tools;
            }
            
            for (Tool tool : allTools) {
                if (tool.getName() != null) {
                    tools.put(tool.getName(), tool);
                }
            }
        } catch (Exception e) {
            log.error("[工具初始化] 从数据库加载工具记录时出错", e);
        }
        
        return tools;
    }
    
    /**
     * 对比Spring容器中的工具Bean和数据库中的工具记录，执行同步操作
     * 
     * 同步策略:
     * 1. Bean存在且数据库已有 → 更新beanName（确保映射正确）
     * 2. Bean存在但数据库无记录 → 创建新的tool记录
     * 3. Bean不存在但数据库有记录 → 记录警告（需要管理员处理）
     * 
     * @param discoveredTools Spring容器中发现的工具Bean
     * @param databaseTools 数据库中的工具记录
     */
    private void synchronizeToolsWithDatabase(Map<String, ToolBeanInfo> discoveredTools, 
                                             Map<String, Tool> databaseTools) {
        int updated = 0;
        int created = 0;
        int skipped = 0;
        int warnings = 0;
        List<String> summaryLog = new ArrayList<>();
        
        // 第一部分：处理发现的Bean
        log.info("[工具同步] 开始处理发现的{}个工具Bean", discoveredTools.size());
        
        for (Map.Entry<String, ToolBeanInfo> entry : discoveredTools.entrySet()) {
            String beanName = entry.getKey();
            ToolBeanInfo beanInfo = entry.getValue();
            String toolName = beanInfo.getDerivedToolName();
            
            try {
                Tool existingTool = databaseTools.get(toolName);
                
                if (existingTool != null) {
                    // 情况1: Bean存在且数据库已有 → 更新beanName
                    if (!beanName.equals(existingTool.getBeanName())) {
                        log.info("[工具同步] 工具'{}': 更新beanName '{}' -> '{}'", 
                            toolName, existingTool.getBeanName(), beanName);
                        existingTool.setBeanName(beanName);
                        toolRepository.updateById(existingTool);
                        updated++;
                        summaryLog.add("✓ 工具'" + toolName + "' beanName已更新为'" + beanName + "'");
                    } else {
                        log.debug("[工具同步] 工具'{}': beanName已是'{}'", toolName, beanName);
                        skipped++;
                        summaryLog.add("- 工具'" + toolName + "' 配置已正确");
                    }
                } else {
                    // 情况2: Bean存在但数据库无记录 → 创建新的tool记录
                    Tool newTool = createToolFromBeanInfo(beanInfo);
                    toolRepository.insert(newTool);
                    created++;
                    log.info("[工具同步] 工具'{}': 创建新的tool记录，beanName为'{}'", toolName, beanName);
                    summaryLog.add("+ 工具'" + toolName + "' 已创建新记录");
                }
            } catch (Exception e) {
                log.error("[工具同步] 处理工具Bean'{}' 时出错: {}", toolName, e.getMessage(), e);
                summaryLog.add("✗ 工具'" + toolName + "' 处理失败: " + e.getMessage());
            }
        }
        
        // 第二部分：检查数据库中的工具是否在Spring容器中存在
        log.info("[工具同步] 开始检查数据库中未匹配的工具");
        
        for (Map.Entry<String, Tool> entry : databaseTools.entrySet()) {
            String toolName = entry.getKey();
            Tool dbTool = entry.getValue();
            
            // 检查这个tool是否对应某个已发现的Bean
            boolean found = discoveredTools.values().stream()
                .anyMatch(beanInfo -> toolName.equalsIgnoreCase(beanInfo.getDerivedToolName()));
            
            if (!found) {
                // 没有对应的Bean，但有beanName配置，验证这个beanName是否有效
                if (dbTool.getBeanName() != null && !dbTool.getBeanName().isEmpty()) {
                    try {
                        Object bean = applicationContext.getBean(dbTool.getBeanName());
                        log.debug("[工具同步] 工具'{}': beanName'{}' 存在但未被发现（可能是特殊配置）", 
                            toolName, dbTool.getBeanName());
                        skipped++;
                    } catch (Exception e) {
                        log.warn("[工具同步] 工具'{}': beanName'{}' 无效，Bean不存在", 
                            toolName, dbTool.getBeanName());
                        summaryLog.add("⚠ 工具'" + toolName + "' beanName'" + dbTool.getBeanName() + "' 无效");
                        warnings++;
                    }
                } else {
                    log.warn("[工具同步] 工具'{}': 数据库记录存在但无对应Bean", toolName);
                    summaryLog.add("⚠ 工具'" + toolName + "' 无对应Bean");
                    warnings++;
                }
            }
        }
        
        // 输出同步总结
        log.info("========== [工具同步] 完成总结 ==========");
        log.info("新建: {} 个，更新: {} 个，跳过: {} 个，警告: {} 个", created, updated, skipped, warnings);
        
        if (!summaryLog.isEmpty()) {
            log.info("[工具同步] 详细信息:");
            summaryLog.forEach(msg -> log.info("  {}", msg));
        }
    }
    
    /**
     * 从Bean信息创建Tool对象
     * 
     * @param beanInfo Bean信息
     * @return 新创建的Tool对象
     */
    private Tool createToolFromBeanInfo(ToolBeanInfo beanInfo) {
        Tool tool = new Tool();
        
        // 基本信息
        tool.setName(beanInfo.getDerivedToolName());
        tool.setBeanName(beanInfo.getBeanName());
        tool.setDisplayName(beanInfo.getSimpleClassName());
        
        // 从类名推导分类和描述
        tool.setCategory("system");
        tool.setDescription("Auto-discovered tool from Bean: " + beanInfo.getFullClassName());
        
        // 默认状态
        tool.setStatus("active");
        
        return tool;
    }
    
    /**
     * 工具Bean信息容器类
     */
    @Data
    @AllArgsConstructor
    @NoArgsConstructor
    public static class ToolBeanInfo {
        /** Spring Bean名称 */
        private String beanName;
        
        /** 类名（不含包名） */
        private String simpleClassName;
        
        /** 完整类名（含包名） */
        private String fullClassName;
        
        /** 包名 */
        private String packageName;
        
        /** Bean的Class对象 */
        private Class<?> beanClass;
        
        /** Bean实例 */
        private Object instance;
        
        /** 推导的工具名称 */
        private String derivedToolName;
    }
    
    /**
     * 手动验证并更新工具的beanName（用于调试和运维）
     * 
     * @param toolId 工具ID
     * @param beanName 待验证的bean名称
     * @return 验证结果
     */
    public boolean validateAndUpdateBeanName(String toolId, String beanName) {
        try {
            log.info("[手动验证] 验证工具{}的beanName'{}'", toolId, beanName);
            
            // 1. 验证bean是否存在
            Object bean = applicationContext.getBean(beanName);
            if (bean == null) {
                log.warn("[手动验证] Bean'{}' 不存在或为null", beanName);
                return false;
            }
            
            // 2. 更新Tool的beanName
            Tool tool = toolRepository.selectById(toolId);
            if (tool == null) {
                log.warn("[手动验证] 工具{}不存在", toolId);
                return false;
            }
            
            tool.setBeanName(beanName);
            toolRepository.updateById(tool);
            log.info("[手动验证] 工具{}的beanName已更新为'{}'", toolId, beanName);
            
            return true;
        } catch (Exception e) {
            log.error("[手动验证] 验证beanName时发生错误", e);
            return false;
        }
    }
}
