package pangea.hiagent.agent.processor;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import pangea.hiagent.model.Agent;
import pangea.hiagent.rag.RagService;
import pangea.hiagent.agent.service.AgentErrorHandler;
import pangea.hiagent.agent.service.TokenConsumerWithCompletion;
import java.util.function.Consumer;

/**
 * Agent处理器抽象基类
 * 封装所有Agent处理器的公共逻辑
 * 职责：提供所有Agent处理器共享的基础功能
 */
@Slf4j
public abstract class AbstractAgentProcessor extends BaseAgentProcessor {

    @Autowired
    protected AgentErrorHandler agentErrorHandler;

    /**
     * 处理RAG响应的通用逻辑
     * 
     * @param ragResponse RAG响应
     * @param tokenConsumer token消费者（流式处理时使用）
     * @return RAG响应
     */
    protected String handleRagResponse(String ragResponse, Consumer<String> tokenConsumer) {
        if (tokenConsumer != null) {
            // 对于流式处理，我们需要将RAG响应作为token发送
            tokenConsumer.accept(ragResponse);
            // 发送完成信号
            if (tokenConsumer instanceof TokenConsumerWithCompletion) {
                ((TokenConsumerWithCompletion) tokenConsumer).onComplete(ragResponse);
            }
        }
        return ragResponse;
    }

    /**
     * 处理请求的通用前置逻辑
     * 
     * @param agent Agent对象
     * @param userMessage 用户消息
     * @param userId 用户ID
     * @param ragService RAG服务
     * @param tokenConsumer token消费者（流式处理时使用）
     * @return RAG响应，如果有的话；否则返回null继续正常处理流程
     */
    protected String handlePreProcessing(Agent agent, String userMessage, String userId, RagService ragService, Consumer<String> tokenConsumer) {
        // 为每个用户-Agent组合创建唯一的会话ID
        String sessionId = generateSessionId(agent, userId);

        // 添加用户消息到ChatMemory
        addUserMessageToMemory(sessionId, userMessage);

        // 检查是否启用RAG并尝试RAG增强
        String ragResponse = tryRagEnhancement(agent, userMessage, ragService);
        if (ragResponse != null) {
            log.info("RAG增强返回结果，直接返回");
            return handleRagResponse(ragResponse, tokenConsumer);
        }

        return null;
    }
}