package pangea.hiagent.web.controller;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import pangea.hiagent.model.Document;
import pangea.hiagent.web.dto.ApiResponse;
import pangea.hiagent.web.service.DocumentService;

import java.util.List;

/**
 * 文档API控制器
 * 提供文档的增删改查功能
 */
@Slf4j
@RestController
@RequestMapping("/api/v1/documents")
@Tag(name = "文档管理", description = "文档管理相关API")
public class DocumentController {
    
    private final DocumentService documentService;
    
    public DocumentController(DocumentService documentService) {
        this.documentService = documentService;
    }
    
    /**
     * 获取所有文档列表
     */
    @GetMapping
    @Operation(summary = "获取文档列表", description = "获取所有文档列表")
    public ApiResponse<List<Document>> getAllDocuments() {
        try {
            log.info("开始获取文档列表");
            List<Document> documents = documentService.getAllDocuments();
            log.info("成功获取文档列表，共{}条记录", documents.size());
            return ApiResponse.success(documents, "获取文档列表成功");
        } catch (Exception e) {
            log.error("获取文档列表失败", e);
            return ApiResponse.error(4001, "获取文档列表失败: " + e.getMessage());
        }
    }
    
    /**
     * 根据ID获取文档
     */
    @GetMapping("/{id}")
    @Operation(summary = "获取文档详情", description = "根据ID获取文档详情")
    public ApiResponse<Document> getDocumentById(@PathVariable String id) {
        try {
            log.info("开始获取文档详情，ID: {}", id);
            Document document = documentService.getDocumentById(id);
            if (document != null) {
                log.info("成功获取文档详情，ID: {}", id);
                return ApiResponse.success(document, "获取文档详情成功");
            } else {
                log.warn("文档不存在，ID: {}", id);
                return ApiResponse.error(4004, "文档不存在");
            }
        } catch (Exception e) {
            log.error("获取文档详情失败，ID: {}", id, e);
            return ApiResponse.error(4001, "获取文档详情失败: " + e.getMessage());
        }
    }
    
    /**
     * 创建文档
     */
    @PostMapping
    @Operation(summary = "创建文档", description = "创建一个新的文档")
    public ApiResponse<Document> createDocument(@RequestBody Document document) {
        try {
            log.info("开始创建文档: {}", document.getName());
            Document created = documentService.createDocument(document);
            log.info("成功创建文档，ID: {}", created.getId());
            return ApiResponse.success(created, "创建文档成功");
        } catch (Exception e) {
            log.error("创建文档失败", e);
            return ApiResponse.error(4001, "创建文档失败: " + e.getMessage());
        }
    }
    
    /**
     * 更新文档
     */
    @PutMapping("/{id}")
    @Operation(summary = "更新文档", description = "更新文档信息")
    public ApiResponse<Document> updateDocument(@PathVariable String id, @RequestBody Document document) {
        try {
            log.info("开始更新文档，ID: {}", id);
            document.setId(id);
            Document updated = documentService.updateDocument(document);
            log.info("成功更新文档，ID: {}", id);
            return ApiResponse.success(updated, "更新文档成功");
        } catch (Exception e) {
            log.error("更新文档失败，ID: {}", id, e);
            return ApiResponse.error(4001, "更新文档失败: " + e.getMessage());
        }
    }
    
    /**
     * 删除文档
     */
    @DeleteMapping("/{id}")
    @Operation(summary = "删除文档", description = "根据ID删除文档")
    public ApiResponse<Void> deleteDocument(@PathVariable String id) {
        try {
            log.info("开始删除文档，ID: {}", id);
            documentService.deleteDocument(id);
            log.info("成功删除文档，ID: {}", id);
            return ApiResponse.success(null, "删除文档成功");
        } catch (Exception e) {
            log.error("删除文档失败，ID: {}", id, e);
            return ApiResponse.error(4001, "删除文档失败: " + e.getMessage());
        }
    }
    
    /**
     * 上传文档
     */
    @PostMapping("/upload")
    @Operation(summary = "上传文档", description = "上传一个新的文档文件")
    public ApiResponse<Document> uploadDocument(@RequestParam("file") MultipartFile file) {
        try {
            log.info("开始上传文档: {}", file.getOriginalFilename());
            
            // 使用服务上传文档
            Document saved = documentService.uploadDocument(file);
            
            log.info("文档上传成功，ID: {}", saved.getId());
            return ApiResponse.success(saved, "文档上传成功");
        } catch (Exception e) {
            log.error("文档上传失败", e);
            return ApiResponse.error(4001, "文档上传失败: " + e.getMessage());
        }
    }
}