import { defineStore } from 'pinia'
import { ref } from 'vue'
import axios, { 
  AxiosInstance, 
  AxiosRequestConfig, 
  AxiosResponse, 
  InternalAxiosRequestConfig, 
  AxiosError 
} from 'axios'
import { useRouter } from 'vue-router'

// 定义用户信息接口
interface UserInfo {
  id?: string
  username?: string
  email?: string
  [key: string]: any
}

// 定义认证响应数据接口
interface AuthResponseData {
  token: string
  [key: string]: any
}

// 定义API响应接口
interface ApiResponse<T = any> {
  code: number
  message: string
  data: T
}


export const useAuthStore = defineStore('auth', () => {
  // 安全地从localStorage获取token
  const getTokenFromStorage = (): string | null => {
    try {
      return localStorage.getItem('token')
    } catch (error) {
      console.warn('无法从localStorage获取token:', error)
      return null
    }
  }

  // 安全地从localStorage获取用户信息
  const getUserInfoFromStorage = (): UserInfo => {
    try {
      const userInfoStr = localStorage.getItem('userInfo')
      return userInfoStr ? JSON.parse(userInfoStr) : {}
    } catch (error) {
      console.warn('无法从localStorage获取用户信息:', error)
      return {}
    }
  }

  const token = ref<string | null>(getTokenFromStorage())
  const userInfo = ref<UserInfo>(getUserInfoFromStorage())
  
  const api: AxiosInstance = axios.create({
    baseURL: '/api/v1'
  })
  
  api.interceptors.request.use(
    (config: InternalAxiosRequestConfig) => {
      if (token.value) {
        config.headers.Authorization = `Bearer ${token.value}`
      }
      return config
    },
    (error: unknown) => {
      console.error('请求拦截器错误:', error)
      return Promise.reject(error)
    }
  )
  
  // 添加响应拦截器处理401错误
  api.interceptors.response.use(
    (response: AxiosResponse) => response,
    (error: unknown) => {
      console.error('响应拦截器错误:', error)
      
      if (axios.isAxiosError(error) && error.response && error.response.status === 401) {
        // 记录详细的错误日志
        console.warn('认证失效，正在清除认证信息并跳转到登录页')
        
        // 清除认证信息
        token.value = null
        userInfo.value = {}
        localStorage.removeItem('token')
        localStorage.removeItem('userInfo')
        
        // 跳转到登录页
        if (typeof window !== 'undefined') {
          setTimeout(() => {
            window.location.href = '/login'
          }, 100)
        }
      }
      return Promise.reject(error)
    }
  )
  
  /**
   * 用户注册
   * @param username 用户名
   * @param password 密码
   * @param email 邮箱
   * @returns 注册响应数据
   */
  async function register(username: string, password: string, email: string): Promise<ApiResponse<AuthResponseData>> {
    try {
      const response: AxiosResponse<ApiResponse<AuthResponseData>> = await api.post('/auth/register', { username, password, email })
      return response.data
    } catch (error: unknown) {
      console.error('注册失败:', error)
      
      if (axios.isAxiosError(error)) {
        const axiosError = error as AxiosError<ApiResponse<null>>
        const errorMessage = axiosError.response?.data?.message || '注册失败'
        console.error('注册错误详情:', errorMessage)
        throw new Error(errorMessage)
      }
      console.error('注册未知错误:', error)
      throw new Error('注册失败')
    }
  }
  
  /**
   * 用户登录
   * @param username 用户名
   * @param password 密码
   * @returns 登录响应数据
   */
  async function login(username: string, password: string): Promise<ApiResponse<AuthResponseData>> {
    try {
      const response: AxiosResponse<ApiResponse<AuthResponseData>> = await api.post('/auth/login', { username, password })
      const { token: newToken } = response.data.data
      token.value = newToken
      localStorage.setItem('token', newToken)
      return response.data
    } catch (error: unknown) {
      console.error('登录失败:', error)
      
      if (axios.isAxiosError(error)) {
        const axiosError = error as AxiosError<ApiResponse<null>>
        const errorMessage = axiosError.response?.data?.message || '登录失败'
        console.error('登录错误详情:', errorMessage)
        throw new Error(errorMessage)
      }
      console.error('登录未知错误:', error)
      throw new Error('登录失败')
    }
  }
  
  /**
   * OAuth2 授权流程
   * 重定向用户到授权服务器
   * @param providerName OAuth2提供商名称
   */
  async function loginWithOAuth2(providerName: string): Promise<void> {
    try {
      // 获取OAuth2授权URL
      const response: AxiosResponse<ApiResponse<{ authorizationUrl: string }>> = await api.get(`/auth/oauth2/authorize?providerName=${providerName}`)
      
      // 重定向到授权URL
      if (response.data.data?.authorizationUrl) {
        window.location.href = response.data.data.authorizationUrl
      } else {
        throw new Error('无法获取OAuth2授权URL')
      }
    } catch (error: unknown) {
      console.error('OAuth2授权失败:', error)
      
      if (axios.isAxiosError(error)) {
        const axiosError = error as AxiosError<ApiResponse<null>>
        const errorMessage = axiosError.response?.data?.message || 'OAuth2授权失败'
        console.error('OAuth2授权错误详情:', errorMessage)
        throw new Error(errorMessage)
      }
      console.error('OAuth2授权未知错误:', error)
      throw new Error('OAuth2授权失败')
    }
  }
  
  /**
   * OAuth2 回调函数
   * 处理来自授权服务器的回调
   * @returns 处理结果Promise
   */
  async function handleOAuth2Callback(): Promise<ApiResponse<AuthResponseData>> {
    // 从 URL 参数中提取授权码
    const params = new URLSearchParams(window.location.search)
    const authCode = params.get('code')
    const providerName = params.get('provider')
    const error = params.get('error')
    
    if (error) {
      const errorMsg = `OAuth2 错误: ${error}`
      console.error(errorMsg)
      throw new Error(errorMsg)
    }
    
    if (!authCode || !providerName) {
      const errorMsg = '不完整的 OAuth2 回调参数'
      console.error(errorMsg)
      throw new Error(errorMsg)
    }
    
    // 使用授权码进行令牌交换
    return await loginWithOAuth2Code(authCode, providerName)
  }
  
  /**
   * 使用授权码进行令牌交换
   * @param authCode 授权码
   * @param providerName 提供商名称
   */
  async function loginWithOAuth2Code(authCode: string, providerName: string): Promise<ApiResponse<AuthResponseData>> {
    try {
      const response: AxiosResponse<ApiResponse<AuthResponseData>> = await api.post('/auth/oauth2/token', {
        authorizationCode: authCode,
        providerName: providerName
      })
      
      const { token: newToken } = response.data.data
      token.value = newToken
      localStorage.setItem('token', newToken)
      
      return response.data
    } catch (error: unknown) {
      if (axios.isAxiosError(error)) {
        const axiosError = error as AxiosError<ApiResponse<null>>
        throw new Error(axiosError.response?.data?.message || 'OAuth2 认证失败')
      }
      throw new Error('OAuth2 认证失败')
    }
  }
  
  /**
   * 用户登出
   */
  function logout(): void {
    token.value = null
    userInfo.value = {}
    try {
      localStorage.removeItem('token')
      localStorage.removeItem('userInfo')
    } catch (error) {
      console.warn('清除localStorage失败:', error)
    }
  }
  
  /**
   * 设置用户信息
   * @param info 用户信息
   */
  function setUserInfo(info: UserInfo): void {
    userInfo.value = info
    try {
      localStorage.setItem('userInfo', JSON.stringify(info))
    } catch (error) {
      console.warn('保存用户信息到localStorage失败:', error)
    }
  }
  
  // 添加便捷方法
  const get = <T = any>(url: string, config: AxiosRequestConfig = {}): Promise<AxiosResponse<T>> => api({ method: 'get', url, ...config })
  const post = <T = any>(url: string, data?: any, config: AxiosRequestConfig = {}): Promise<AxiosResponse<T>> => api({ method: 'post', url, data, ...config })
  const put = <T = any>(url: string, data?: any, config: AxiosRequestConfig = {}): Promise<AxiosResponse<T>> => api({ method: 'put', url, data, ...config })
  const del = <T = any>(url: string, config: AxiosRequestConfig = {}): Promise<AxiosResponse<T>> => api({ method: 'delete', url, ...config })
  
  return {
    token,
    userInfo,
    register,
    login,
    loginWithOAuth2,
    handleOAuth2Callback,
    loginWithOAuth2Code,
    logout,
    setUserInfo,
    api,
    get,
    post,
    put,
    del
  }
})