package pangea.hiagent.tool.impl;

import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.tool.annotation.Tool;
import org.springframework.stereotype.Component;

/**
 * 文档模板工具
 * 用于提供各种类型的文档模板
 */
@Slf4j
@Component
public class DocumentTemplateTool {
    
    public DocumentTemplateTool() {
        // 默认构造器
    }
    
    /**
     * 获取文档模板
     * @param templateType 模板类型（如报告、提案、邮件等）
     * @param industry 行业领域（如科技、金融、教育等）
     * @return 文档模板内容
     */
    @Tool(description = "根据模板类型和行业领域提供文档模板")
    public String getDocumentTemplate(String templateType, String industry) {
        log.debug("开始获取文档模板: 模板类型={}, 行业领域={}", templateType, industry);
        
        try {
            if (templateType == null || templateType.trim().isEmpty()) {
                log.warn("模板类型不能为空");
                return "错误：模板类型不能为空";
            }
            
            // 生成文档模板
            String template = generateDocumentTemplate(templateType, industry);
            
            log.info("文档模板生成完成: 模板类型={}, 行业领域={}", templateType, industry);
            return template;
        } catch (Exception e) {
            log.error("获取文档模板时发生错误: {}", e.getMessage(), e);
            return "获取文档模板时发生错误: " + e.getMessage();
        }
    }
    
    /**
     * 生成文档模板
     * @param templateType 模板类型
     * @param industry 行业领域
     * @return 文档模板内容
     */
    private String generateDocumentTemplate(String templateType, String industry) {
        StringBuilder template = new StringBuilder();
        template.append("文档模板:\n\n");
        template.append("模板类型: ").append(templateType).append("\n");
        if (industry != null && !industry.trim().isEmpty()) {
            template.append("行业领域: ").append(industry).append("\n");
        }
        template.append("\n");
        
        // 根据模板类型生成具体内容
        switch (templateType.toLowerCase()) {
            case "报告":
                template.append("【报告标题】\n\n");
                template.append("1. 摘要\n");
                template.append("   - 简要概述报告的主要内容和结论\n\n");
                template.append("2. 引言\n");
                template.append("   - 背景介绍\n");
                template.append("   - 报告目的\n");
                template.append("   - 研究范围\n\n");
                template.append("3. 主体内容\n");
                template.append("   - 数据分析\n");
                template.append("   - 结果展示\n");
                template.append("   - 问题识别\n\n");
                template.append("4. 结论与建议\n");
                template.append("   - 主要发现\n");
                template.append("   - 改进建议\n");
                template.append("   - 后续步骤\n\n");
                template.append("5. 附录\n");
                template.append("   - 数据表格\n");
                template.append("   - 参考文献\n");
                break;
                
            case "提案":
                template.append("【提案标题】\n\n");
                template.append("1. 执行摘要\n");
                template.append("   - 提案核心内容概述\n");
                template.append("   - 预期收益\n\n");
                template.append("2. 背景与现状\n");
                template.append("   - 当前情况分析\n");
                template.append("   - 存在的问题\n\n");
                template.append("3. 解决方案\n");
                template.append("   - 具体措施\n");
                template.append("   - 实施步骤\n");
                template.append("   - 时间安排\n\n");
                template.append("4. 预算与资源\n");
                template.append("   - 成本估算\n");
                template.append("   - 所需资源\n\n");
                template.append("5. 风险评估\n");
                template.append("   - 潜在风险\n");
                template.append("   - 应对策略\n\n");
                template.append("6. 结论\n");
                template.append("   - 总结要点\n");
                template.append("   - 呼吁行动\n");
                break;
                
            case "邮件":
                template.append("主题: [简明扼要地概括邮件内容]\n\n");
                template.append("尊敬的[收件人姓名]:\n\n");
                template.append("[开场白 - 简要说明写信目的]\n\n");
                template.append("[主体内容 - 详细阐述相关信息]\n\n");
                template.append("[结尾 - 总结要点，提出下一步行动建议]\n\n");
                template.append("此致\n敬礼!\n\n");
                template.append("[发件人姓名]\n");
                template.append("[职位]\n");
                template.append("[联系方式]\n");
                break;
                
            case "说明书":
                template.append("【产品名称】使用说明书\n\n");
                template.append("1. 产品简介\n");
                template.append("   - 产品功能\n");
                template.append("   - 适用范围\n\n");
                template.append("2. 安全须知\n");
                template.append("   - 使用前注意事项\n");
                template.append("   - 安全警告\n\n");
                template.append("3. 安装指南\n");
                template.append("   - 所需工具\n");
                template.append("   - 安装步骤\n");
                template.append("   - 图示说明\n\n");
                template.append("4. 使用方法\n");
                template.append("   - 操作步骤\n");
                template.append("   - 功能说明\n\n");
                template.append("5. 维护保养\n");
                template.append("   - 日常维护\n");
                template.append("   - 故障排除\n\n");
                template.append("6. 技术参数\n");
                template.append("   - 规格参数\n");
                template.append("   - 性能指标\n\n");
                template.append("7. 售后服务\n");
                template.append("   - 保修条款\n");
                template.append("   - 联系方式\n");
                break;
                
            default:
                template.append("【文档标题】\n\n");
                template.append("1. 章节一\n");
                template.append("   - 内容要点1\n");
                template.append("   - 内容要点2\n\n");
                template.append("2. 章节二\n");
                template.append("   - 内容要点1\n");
                template.append("   - 内容要点2\n\n");
                template.append("3. 章节三\n");
                template.append("   - 内容要点1\n");
                template.append("   - 内容要点2\n\n");
                template.append("4. 结论\n");
                template.append("   - 总结要点\n");
                template.append("   - 后续建议\n");
                break;
        }
        
        return template.toString();
    }
}