package pangea.hiagent.tool.impl;

import com.fasterxml.jackson.annotation.JsonClassDescription;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonPropertyDescription;
import jakarta.mail.*;
import jakarta.mail.internet.*;
import jakarta.mail.search.FlagTerm;
import jakarta.mail.search.ReceivedDateTerm;
import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.tool.annotation.Tool;
import org.springframework.stereotype.Component;


import java.io.File;
import java.util.*;

/**
 * 邮件工具类
 * 提供基于POP3协议的邮件访问功能
 */
@Slf4j
@Component
public class EmailTools {
    
    private Boolean pop3SslEnable = true;
    
    private String pop3SocketFactoryClass = "javax.net.ssl.SSLSocketFactory";

    // 邮件请求参数类
    @JsonClassDescription("邮件操作请求参数")
    public record EmailRequest(
            @JsonProperty(required = true, value = "host")
            @JsonPropertyDescription("POP3服务器地址")
            String host,
            
            @JsonProperty(value = "port")
            @JsonPropertyDescription("POP3服务器端口，默认995")
            Integer port,
            
            @JsonProperty(required = true, value = "username")
            @JsonPropertyDescription("邮箱用户名")
            String username,
            
            @JsonProperty(required = true, value = "password")
            @JsonPropertyDescription("邮箱密码")
            String password,
            
            @JsonProperty(value = "mailId")
            @JsonPropertyDescription("邮件ID，用于指定操作的邮件")
            Integer mailId,
            
            @JsonProperty(value = "attachmentPath")
            @JsonPropertyDescription("附件保存路径")
            String attachmentPath
    ) {
        // 默认端口为配置值
        public Integer port() {
            return port != null ? port : 995;
        }
    }

    // 邮件基本信息响应类
    @JsonClassDescription("邮件基本信息响应")
    public record EmailBasicInfo(
            @JsonPropertyDescription("邮件ID") Integer mailId,
            @JsonPropertyDescription("发件人") String from,
            @JsonPropertyDescription("收件人") String to,
            @JsonPropertyDescription("邮件标题") String subject,
            @JsonPropertyDescription("接收日期") String receivedDate,
            @JsonPropertyDescription("是否已读") boolean isRead
    ) {}

    // 邮件详细信息响应类
    @JsonClassDescription("邮件详细信息响应")
    public record EmailDetailInfo(
            @JsonPropertyDescription("邮件ID") Integer mailId,
            @JsonPropertyDescription("发件人") String from,
            @JsonPropertyDescription("收件人") String to,
            @JsonPropertyDescription("抄送") String cc,
            @JsonPropertyDescription("密送") String bcc,
            @JsonPropertyDescription("邮件标题") String subject,
            @JsonPropertyDescription("接收日期") String receivedDate,
            @JsonPropertyDescription("是否已读") boolean isRead,
            @JsonPropertyDescription("邮件内容") String content,
            @JsonPropertyDescription("附件列表") List<EmailAttachment> attachments
    ) {}

    // 邮件附件信息类
    @JsonClassDescription("邮件附件信息")
    public record EmailAttachment(
            @JsonPropertyDescription("附件名称") String filename,
            @JsonPropertyDescription("附件大小") long size,
            @JsonPropertyDescription("内容类型") String contentType,
            @JsonPropertyDescription("保存路径") String savePath
    ) {}

    // 操作结果响应类
    @JsonClassDescription("邮件操作结果响应")
    public record EmailOperationResult(
            @JsonPropertyDescription("操作是否成功") boolean success,
            @JsonPropertyDescription("操作结果消息") String message,
            @JsonPropertyDescription("附加数据") Map<String, Object> data
    ) {}

    /**
     * 获取今日所有邮件的发件人和标题
     * @param request 邮件请求参数
     * @return 今日邮件列表
     */
    @Tool(description = "获取今日所有邮件的发件人和标题")
    public List<EmailBasicInfo> getTodayEmails(EmailRequest request) {
        log.debug("获取今日邮件，请求参数：{}", request);
        try {
            Store store = connectToEmailServer(request);
            Folder inbox = store.getFolder("INBOX");
            inbox.open(Folder.READ_ONLY);

            // 获取今日日期范围
            Calendar today = Calendar.getInstance();
            today.set(Calendar.HOUR_OF_DAY, 0);
            today.set(Calendar.MINUTE, 0);
            today.set(Calendar.SECOND, 0);
            today.set(Calendar.MILLISECOND, 0);

            Calendar tomorrow = Calendar.getInstance();
            tomorrow.setTime(today.getTime());
            tomorrow.add(Calendar.DAY_OF_MONTH, 1);

            // 搜索今日邮件
            ReceivedDateTerm term = new ReceivedDateTerm(ReceivedDateTerm.GE, today.getTime());
            Message[] messages = inbox.search(term);

            // 过滤出今日邮件并转换为响应格式
            List<EmailBasicInfo> result = new ArrayList<>();
            for (int i = 0; i < messages.length; i++) {
                Message message = messages[i];
                Calendar receivedDate = Calendar.getInstance();
                receivedDate.setTime(message.getReceivedDate());
                
                // 只保留今日邮件
                if (receivedDate.before(tomorrow)) {
                    result.add(convertToBasicInfo(i + 1, message));
                }
            }

            inbox.close(false);
            store.close();
            log.debug("获取今日邮件成功，共{}封", result.size());
            return result;
        } catch (Exception e) {
            log.error("获取今日邮件失败：", e);
            throw new RuntimeException("获取今日邮件失败：" + e.getMessage());
        }
    }

    /**
     * 获取所有未读邮件的发件人和标题
     * @param request 邮件请求参数
     * @return 未读邮件列表
     */
    @Tool(description = "获取所有未读邮件的发件人和标题")
    public List<EmailBasicInfo> getUnreadEmails(EmailRequest request) {
        log.debug("获取未读邮件，请求参数：{}", request);
        try {
            Store store = connectToEmailServer(request);
            Folder inbox = store.getFolder("INBOX");
            inbox.open(Folder.READ_ONLY);

            // 搜索未读邮件
            FlagTerm term = new FlagTerm(new Flags(Flags.Flag.SEEN), false);
            Message[] messages = inbox.search(term);

            // 转换为响应格式
            List<EmailBasicInfo> result = new ArrayList<>();
            for (int i = 0; i < messages.length; i++) {
                result.add(convertToBasicInfo(i + 1, messages[i]));
            }

            inbox.close(false);
            store.close();
            log.debug("获取未读邮件成功，共{}封", result.size());
            return result;
        } catch (Exception e) {
            log.error("获取未读邮件失败：", e);
            throw new RuntimeException("获取未读邮件失败：" + e.getMessage());
        }
    }

    /**
     * 获取指定邮件的内容
     * @param request 邮件请求参数，必须包含mailId
     * @return 邮件详细信息
     */
    @Tool(description = "获取指定邮件的内容")
    public EmailDetailInfo getEmailContent(EmailRequest request) {
        log.debug("获取邮件内容，请求参数：{}", request);
        if (request.mailId() == null) {
            throw new IllegalArgumentException("邮件ID不能为空");
        }

        try {
            Store store = connectToEmailServer(request);
            Folder inbox = store.getFolder("INBOX");
            inbox.open(Folder.READ_WRITE);

            int messageCount = inbox.getMessageCount();
            if (request.mailId() < 1 || request.mailId() > messageCount) {
                throw new IllegalArgumentException("邮件ID无效：" + request.mailId());
            }

            Message message = inbox.getMessage(request.mailId());
            EmailDetailInfo result = convertToDetailInfo(request.mailId(), message);

            inbox.close(false);
            store.close();
            log.debug("获取邮件内容成功，邮件ID：{}", request.mailId());
            return result;
        } catch (Exception e) {
            log.error("获取邮件内容失败：", e);
            throw new RuntimeException("获取邮件内容失败：" + e.getMessage());
        }
    }

    /**
     * 获取指定邮件的附件
     * @param request 邮件请求参数，必须包含mailId和attachmentPath
     * @return 附件信息列表
     */
    @Tool(description = "获取指定邮件的附件")
    public List<EmailAttachment> getEmailAttachments(EmailRequest request) {
        log.debug("获取邮件附件，请求参数：{}", request);
        if (request.mailId() == null) {
            throw new IllegalArgumentException("邮件ID不能为空");
        }
        if (request.attachmentPath() == null) {
            throw new IllegalArgumentException("附件保存路径不能为空");
        }

        try {
            Store store = connectToEmailServer(request);
            Folder inbox = store.getFolder("INBOX");
            inbox.open(Folder.READ_ONLY);

            int messageCount = inbox.getMessageCount();
            if (request.mailId() < 1 || request.mailId() > messageCount) {
                throw new IllegalArgumentException("邮件ID无效：" + request.mailId());
            }

            Message message = inbox.getMessage(request.mailId());
            List<EmailAttachment> attachments = saveAttachments(message, request.attachmentPath());

            inbox.close(false);
            store.close();
            log.debug("获取邮件附件成功，邮件ID：{}，共{}个附件", request.mailId(), attachments.size());
            return attachments;
        } catch (Exception e) {
            log.error("获取邮件附件失败：", e);
            throw new RuntimeException("获取邮件附件失败：" + e.getMessage());
        }
    }

    /**
     * 标记邮件为已读
     * @param request 邮件请求参数，必须包含mailId
     * @return 操作结果
     */
    @Tool(description = "标记邮件为已读")
    public EmailOperationResult markAsRead(EmailRequest request) {
        log.debug("标记邮件为已读，请求参数：{}", request);
        return markEmailFlag(request, Flags.Flag.SEEN, true);
    }

    /**
     * 标记邮件为未读
     * @param request 邮件请求参数，必须包含mailId
     * @return 操作结果
     */
    @Tool(description = "标记邮件为未读")
    public EmailOperationResult markAsUnread(EmailRequest request) {
        log.debug("标记邮件为未读，请求参数：{}", request);
        return markEmailFlag(request, Flags.Flag.SEEN, false);
    }

    /**
     * 删除指定邮件
     * @param request 邮件请求参数，必须包含mailId
     * @return 操作结果
     */
    @Tool(description = "删除指定邮件")
    public EmailOperationResult deleteEmail(EmailRequest request) {
        log.debug("删除邮件，请求参数：{}", request);
        if (request.mailId() == null) {
            throw new IllegalArgumentException("邮件ID不能为空");
        }

        try {
            Store store = connectToEmailServer(request);
            Folder inbox = store.getFolder("INBOX");
            inbox.open(Folder.READ_WRITE);

            int messageCount = inbox.getMessageCount();
            if (request.mailId() < 1 || request.mailId() > messageCount) {
                throw new IllegalArgumentException("邮件ID无效：" + request.mailId());
            }

            Message message = inbox.getMessage(request.mailId());
            message.setFlag(Flags.Flag.DELETED, true);

            inbox.close(true); // 提交删除操作
            store.close();
            log.debug("删除邮件成功，邮件ID：{}", request.mailId());
            
            Map<String, Object> data = new HashMap<>();
            data.put("mailId", request.mailId());
            return new EmailOperationResult(true, "邮件删除成功", data);
        } catch (Exception e) {
            log.error("删除邮件失败：", e);
            return new EmailOperationResult(false, "邮件删除失败：" + e.getMessage(), Collections.emptyMap());
        }
    }

    // 连接到邮件服务器
    private Store connectToEmailServer(EmailRequest request) throws Exception {
        Properties props = new Properties();
        props.put("mail.pop3.host", request.host());
        props.put("mail.pop3.port", request.port().toString());
        props.put("mail.pop3.ssl.enable", pop3SslEnable.toString());
        props.put("mail.pop3.socketFactory.class", pop3SocketFactoryClass);
        props.put("mail.pop3.socketFactory.port", request.port().toString());

        Session session = Session.getInstance(props, null);
        Store store = session.getStore("pop3s");
        store.connect(request.host(), request.username(), request.password());
        log.debug("成功连接到邮件服务器：{}:{}", request.host(), request.port());
        return store;
    }

    // 将Message转换为EmailBasicInfo
    private EmailBasicInfo convertToBasicInfo(int mailId, Message message) throws Exception {
        String from = InternetAddress.toString(message.getFrom());
        String to = InternetAddress.toString(message.getRecipients(Message.RecipientType.TO));
        String subject = message.getSubject() != null ? MimeUtility.decodeText(message.getSubject()) : "无主题";
        String receivedDate = message.getReceivedDate().toString();
        boolean isRead = message.isSet(Flags.Flag.SEEN);

        return new EmailBasicInfo(mailId, from, to, subject, receivedDate, isRead);
    }

    // 将Message转换为EmailDetailInfo
    private EmailDetailInfo convertToDetailInfo(int mailId, Message message) throws Exception {
        String from = InternetAddress.toString(message.getFrom());
        String to = message.getRecipients(Message.RecipientType.TO) != null ? 
                InternetAddress.toString(message.getRecipients(Message.RecipientType.TO)) : "";
        String cc = message.getRecipients(Message.RecipientType.CC) != null ? 
                InternetAddress.toString(message.getRecipients(Message.RecipientType.CC)) : "";
        String bcc = message.getRecipients(Message.RecipientType.BCC) != null ? 
                InternetAddress.toString(message.getRecipients(Message.RecipientType.BCC)) : "";
        String subject = message.getSubject() != null ? MimeUtility.decodeText(message.getSubject()) : "无主题";
        String receivedDate = message.getReceivedDate().toString();
        boolean isRead = message.isSet(Flags.Flag.SEEN);
        String content = getEmailContent(message);

        return new EmailDetailInfo(mailId, from, to, cc, bcc, subject, receivedDate, isRead, content, Collections.emptyList());
    }

    // 获取邮件内容
    private String getEmailContent(Part part) throws Exception {
        if (part.isMimeType("text/*")) {
            return part.getContent().toString();
        } else if (part.isMimeType("multipart/*")) {
            Multipart multipart = (Multipart) part.getContent();
            StringBuilder content = new StringBuilder();
            for (int i = 0; i < multipart.getCount(); i++) {
                BodyPart bodyPart = multipart.getBodyPart(i);
                if (!bodyPart.isMimeType("multipart/*") && !Part.ATTACHMENT.equalsIgnoreCase(bodyPart.getDisposition())) {
                    content.append(getEmailContent(bodyPart));
                }
            }
            return content.toString();
        }
        return "";
    }

    // 保存邮件附件
    private List<EmailAttachment> saveAttachments(Message message, String savePath) throws Exception {
        List<EmailAttachment> attachments = new ArrayList<>();
        
        // 创建保存目录
        File directory = new File(savePath);
        if (!directory.exists()) {
            directory.mkdirs();
        }

        if (message.isMimeType("multipart/*")) {
            Multipart multipart = (Multipart) message.getContent();
            for (int i = 0; i < multipart.getCount(); i++) {
                BodyPart bodyPart = multipart.getBodyPart(i);
                String disposition = bodyPart.getDisposition();
                
                if (disposition != null && (disposition.equalsIgnoreCase(Part.ATTACHMENT) || 
                        disposition.equalsIgnoreCase(Part.INLINE))) {
                    // 处理附件
                    String filename = getFilename(bodyPart);
                    File attachmentFile = new File(savePath + File.separator + filename);
                    
                    // 使用writeTo方法保存附件
                    try (var inputStream = bodyPart.getInputStream();
                         var outputStream = new java.io.FileOutputStream(attachmentFile)) {
                        byte[] buffer = new byte[4096];
                        int bytesRead;
                        while ((bytesRead = inputStream.read(buffer)) != -1) {
                            outputStream.write(buffer, 0, bytesRead);
                        }
                    }
                    
                    EmailAttachment attachment = new EmailAttachment(
                            filename,
                            attachmentFile.length(),
                            bodyPart.getContentType(),
                            attachmentFile.getAbsolutePath()
                    );
                    attachments.add(attachment);
                } else if (bodyPart.isMimeType("multipart/*")) {
                    // 递归处理嵌套的multipart
                    // 创建一个临时Message对象来处理嵌套的multipart
                    MimeMessage nestedMessage = new MimeMessage((Session) null);
                    nestedMessage.setContent(bodyPart.getContent(), bodyPart.getContentType());
                    attachments.addAll(saveAttachments(nestedMessage, savePath));
                }
            }
        }
        
        return attachments;
    }

    // 获取附件文件名
    private String getFilename(BodyPart bodyPart) throws Exception {
        String filename = bodyPart.getFileName();
        if (filename != null) {
            return MimeUtility.decodeText(filename);
        }
        return "attachment_" + System.currentTimeMillis() + ".dat";
    }

    // 标记邮件标志
    private EmailOperationResult markEmailFlag(EmailRequest request, Flags.Flag flag, boolean set) {
        if (request.mailId() == null) {
            throw new IllegalArgumentException("邮件ID不能为空");
        }

        try {
            Store store = connectToEmailServer(request);
            Folder inbox = store.getFolder("INBOX");
            inbox.open(Folder.READ_WRITE);

            int messageCount = inbox.getMessageCount();
            if (request.mailId() < 1 || request.mailId() > messageCount) {
                throw new IllegalArgumentException("邮件ID无效：" + request.mailId());
            }

            Message message = inbox.getMessage(request.mailId());
            message.setFlag(flag, set);

            inbox.close(false);
            store.close();
            
            String action = set ? "标记为已读" : "标记为未读";
            log.debug("邮件{}成功，邮件ID：{}", action, request.mailId());
            
            Map<String, Object> data = new HashMap<>();
            data.put("mailId", request.mailId());
            data.put("isRead", set);
            return new EmailOperationResult(true, "邮件" + action + "成功", data);
        } catch (Exception e) {
            log.error("邮件标记操作失败：", e);
            String action = set ? "标记为已读" : "标记为未读";
            return new EmailOperationResult(false, "邮件" + action + "失败：" + e.getMessage(), Collections.emptyMap());
        }
    }
}