package pangea.hiagent.tool.impl;import lombok.extern.slf4j.Slf4j;
import pangea.hiagent.workpanel.IWorkPanelDataCollector;

import org.springframework.ai.tool.annotation.Tool;
import org.springframework.stereotype.Component;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.net.URI;
import java.net.URL;
import java.net.URLConnection;

/**
 * 网页访问工具类
 * 提供根据网站名称或URL地址访问网页并在工作面板中预览的功能
 */
@Slf4j
@Component
public class WebPageAccessTools {

    // 通过构造器注入的方式引入IWorkPanelDataCollector依赖
    private final IWorkPanelDataCollector workPanelDataCollector;

    public WebPageAccessTools(IWorkPanelDataCollector workPanelDataCollector) {
        this.workPanelDataCollector = workPanelDataCollector;
    }

    /**
     * 根据网站名称访问网页并在工作面板中预览
     * @param siteName 网站名称（如"百度"、"谷歌"等）
     * @return 操作结果描述
     */
    @Tool(description = "根据网站名称访问网页并在工作面板中预览，支持常见网站如百度、谷歌、必应等，如未找到匹配的网站名称则返回错误信息")
    public String accessWebSiteByName(String siteName) {
        log.debug("接收到访问网站请求，网站名称: {}", siteName);
        
        try {
            String url = getWebsiteUrlByName(siteName);
            if (url != null) {
                return accessAndRecordWebContent(url, "网页预览: " + siteName);
            } else {
                return handleFailure("未找到匹配的网站名称: " + siteName);
            }
        } catch (Exception e) {
            return handleError(e, "访问网站时发生错误");
        }
    }
    
    /**
     * 根据URL地址直接访问网页并在工作面板中预览
     * @param url 网页URL地址
     * @return 操作结果描述
     */
    @Tool(description = "根据完整的URL地址访问网页并在工作面板中预览")
    public String accessWebSiteByUrl(String url) {
        log.debug("接收到访问网页URL请求: {}", url);
        
        try {
            // 自动补全URL协议
            String completeUrl = completeUrlProtocol(url);
            
            // 验证URL格式
            if (!isValidUrl(completeUrl)) {
                return handleFailure("无效的URL格式，请确保输入完整的网址，例如: https://www.example.com");
            }
            
            return accessAndRecordWebContent(completeUrl, "网页预览");
        } catch (Exception e) {
            return handleError(e, "访问网页时发生错误");
        }
    }

    /**
     * 访问网页内容并记录到工作面板
     * @param url 网页URL
     * @param title embed标题
     * @return 操作结果
     */
    private String accessAndRecordWebContent(String url, String title) {
        try {
            // 直接获取网页内容并发送到工作面板进行预览，不再打开浏览器
            String webContent = fetchWebContent(url);
            
            log.info("成功访问网页: {}", url);
            String result = "已成功在工作面板中预览网页: " + url;
            
            // 发送embed事件到工作面板
            workPanelDataCollector.recordEmbed(url, "text/html", title, webContent);
            
            return result;
        } catch (Exception e) {
            return handleError(e, "获取网页内容时发生错误");
        }
    }

    /**
     * 处理失败情况
     * @param message 错误消息
     * @return 错误结果
     */
    private String handleFailure(String message) {
        log.warn(message);
        return "抱歉，" + message;
    }

    /**
     * 处理异常情况
     * @param e 异常对象
     * @param prefix 错误消息前缀
     * @return 错误结果
     */
    private String handleError(Exception e, String prefix) {
        log.error("{}: {}, 错误详情: {}", prefix, e.getMessage(), e);
        return prefix + ": " + e.getMessage();
    }

    /**
     * 根据网站名称获取对应的URL
     * @param siteName 网站名称
     * @return 对应的URL，如果未找到则返回null
     */
    private String getWebsiteUrlByName(String siteName) {
        if (siteName == null || siteName.isEmpty()) {
            return null;
        }

        // 转换为小写进行比较
        String lowerSiteName = siteName.toLowerCase();

        // 常见网站映射
        switch (lowerSiteName) {
            case "百度":
            case "baidu":
                return "https://www.baidu.com";
            case "谷歌":
            case "google":
                return "https://www.google.com";
            case "必应":
            case "bing":
                return "https://www.bing.com";
            case "淘宝":
            case "taobao":
                return "https://www.taobao.com";
            case "京东":
            case "jd":
            case "jingdong":
                return "https://www.jd.com";
            case "天猫":
            case "tmall":
                return "https://www.tmall.com";
            case "知乎":
            case "zhihu":
                return "https://www.zhihu.com";
            case "微博":
            case "weibo":
                return "https://www.weibo.com";
            case "github":
                return "https://www.github.com";
            default:
                return null;
        }
    }

    /**
     * 自动补全URL协议
     * @param url 待补全的URL
     * @return 补全后的URL
     */
    private String completeUrlProtocol(String url) {
        if (url == null || url.isEmpty()) {
            return url;
        }
        
        // 如果已经包含协议，则直接返回
        if (url.startsWith("http://") || url.startsWith("https://")) {
            return url;
        }
        
        // 默认添加https协议
        return "https://" + url;
    }

    /**
     * 验证URL格式是否有效
     * @param url 待验证的URL
     * @return 如果URL格式有效返回true，否则返回false
     */
    private boolean isValidUrl(String url) {
        if (url == null || url.isEmpty()) {
            return false;
        }
        
        // 检查是否包含协议
        if (!url.startsWith("http://") && !url.startsWith("https://")) {
            return false;
        }
        
        // 简单的URL格式检查
        try {
            URI uri = URI.create(url);
            return uri.getHost() != null && !uri.getHost().isEmpty();
        } catch (Exception e) {
            return false;
        }
    }
    
    /**
     * 获取网页内容
     * @param url 网页URL地址
     * @return 网页HTML内容
     */
    private String fetchWebContent(String url) {
        StringBuilder content = new StringBuilder();
        BufferedReader reader = null;
        
        try {
            URLConnection connection = new URL(url).openConnection();
            // 设置请求头，模拟浏览器访问
            connection.setRequestProperty("User-Agent", 
                "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36");
            connection.setConnectTimeout(5000); // 5秒连接超时
            connection.setReadTimeout(10000);   // 10秒读取超时
            
            reader = new BufferedReader(new InputStreamReader(connection.getInputStream(), "UTF-8"));
            String line;
            while ((line = reader.readLine()) != null) {
                content.append(line).append("\n");
            }
            
            log.debug("成功获取网页内容，长度: {} bytes", content.length());
            return content.toString();
            
        } catch (Exception e) {
            log.error("获取网页内容失败: {}", e.getMessage(), e);
            // 返回错误提示页面
            return "<html><body><h1>获取网页内容失败</h1><p>错误信息: " + e.getMessage() + "</p></body></html>";
        } finally {
            if (reader != null) {
                try {
                    reader.close();
                } catch (Exception e) {
                    log.debug("关闭读取器失败: {}", e.getMessage());
                }
            }
        }
    }
}