package pangea.hiagent.llm;

import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.chat.model.ChatModel;
import org.springframework.ai.openai.OpenAiChatModel;
import org.springframework.ai.openai.api.OpenAiApi;
import org.springframework.ai.openai.OpenAiChatOptions;
import pangea.hiagent.model.LlmConfig;

/**
 * 抽象OpenAI兼容适配器
 * 封装OpenAI兼容模型的通用创建逻辑
 */
@Slf4j
public abstract class AbstractOpenAiCompatibleAdapter implements ModelAdapter {
    
    /**
     * 创建OpenAI兼容的ChatModel实例
     * 
     * @param config LLM配置
     * @param defaultBaseUrl 默认API基础URL
     * @return ChatModel实例
     */
    protected ChatModel createOpenAiChatModel(LlmConfig config, String defaultBaseUrl) {
        try {
            // 获取或设置基础URL
            String baseUrl = (config.getBaseUrl() != null && !config.getBaseUrl().isEmpty()) ? 
                    config.getBaseUrl() : defaultBaseUrl;
            
            log.debug("创建OpenAI兼容ChatModel, 提供商: {}, baseUrl: {}, modelName: {}", 
                    getProviderName(), baseUrl, config.getModelName());
            
            // 创建OpenAiApi实例
            OpenAiApi openAiApi = OpenAiApi.builder()
                    .apiKey(config.getApiKey())
                    .baseUrl(baseUrl)
                    .build();
            
            // 创建OpenAiChatOptions配置
            OpenAiChatOptions options = OpenAiChatOptions.builder()
                    .model(config.getModelName())
                    .temperature(config.getTemperature())
                    .maxTokens(config.getMaxTokens())
                    .build();
            
            // 创建OpenAiChatModel实例
            OpenAiChatModel chatModel = OpenAiChatModel.builder()
                    .openAiApi(openAiApi)
                    .defaultOptions(options)
                    .build();
            
            log.debug("OpenAI兼容ChatModel创建成功, 提供商: {}", getProviderName());
            return chatModel;
        } catch (Exception e) {
            log.error("创建OpenAI兼容ChatModel失败, 提供商: {}, 错误: {}", getProviderName(), e.getMessage(), e);
            throw new RuntimeException("创建ChatModel失败: " + e.getMessage(), e);
        }
    }
    
    /**
     * 验证OpenAI兼容配置的基本要求
     * 
     * @param config LLM配置
     * @param requireApiKey 是否必须提供API密钥
     * @return 配置是否有效
     */
    protected boolean validateOpenAiConfig(LlmConfig config, boolean requireApiKey) {
        if (config == null || !config.getEnabled()) {
            return false;
        }
        
        if (requireApiKey && (config.getApiKey() == null || config.getApiKey().isEmpty())) {
            log.warn("配置验证失败，提供商: {}, 原因: 缺少API密钥", getProviderName());
            return false;
        }
        
        return true;
    }
}