// 内容展开管理hook
import { nextTick, ref, type Ref } from 'vue'
import type { TimelineEvent } from '../types/timeline'

export function useContentExpansion(props: {
  events: TimelineEvent[]
}) {
  // 内容展开状态管理 - 使用WeakMap提高性能
  const contentExpandedStates = new WeakMap<HTMLElement, boolean>()
  const contentLineCounts = ref<Record<string, number>>({})
  const contentElements = new Map<string, HTMLElement>()
  
  // 事件ID到时间戳的映射，用于快速查找
  const eventIdToTimestamp = ref<Record<string, number>>({})

  // 更新事件ID映射
  const updateEventIdMapping = () => {
    props.events.forEach(event => {
      if (event.id) {
        eventIdToTimestamp.value[event.id] = event.timestamp
      }
    })
  }

  // 获取内容展开状态
  const getContentExpandedState = (timestamp: number): boolean => {
    const key = timestamp.toString()
    const element = contentElements.get(key)
    return element ? (contentExpandedStates.get(element) || false) : false
  }

  // 注册内容元素引用
  const setContentRef = (el: HTMLElement | null, timestamp: number) => {
    if (el) {
      const key = timestamp.toString()
      contentElements.set(key, el)
      // 初始化展开状态为false
      if (!contentExpandedStates.has(el)) {
        contentExpandedStates.set(el, false)
      }
      // 更新行数计算
      updateLineCountForElement(timestamp)
    }
  }

  // 为特定元素更新行数计算
  const updateLineCountForElement = (timestamp: number) => {
    const event = props.events.find(e => e.timestamp === timestamp)
    const key = timestamp.toString()
    const element = contentElements.get(key)
    
    if (event && 'content' in event && event.content && element) {
      const lineCount = calculateLineCount(event.content, element)
      const contentKey = event.id || key
      contentLineCounts.value[contentKey] = lineCount
    }
  }

  // 切换内容展开状态
  const toggleContentExpand = (timestamp: number) => {
    const key = timestamp.toString()
    const element = contentElements.get(key)
    if (element) {
      const currentState = contentExpandedStates.get(element) || false
      contentExpandedStates.set(element, !currentState)
    }
  }

  // 检查是否应该显示切换按钮
  // 计算内容实际显示行数（考虑自动换行）
  const calculateLineCount = (content: string, element?: HTMLElement): number => {
    if (!content) return 0
    
    // 如果有DOM元素，使用更精确的方法计算行数
    if (element) {
      const computedStyle = window.getComputedStyle(element)
      const lineHeight = parseFloat(computedStyle.lineHeight)
      const height = element.scrollHeight // 使用scrollHeight更准确
      
      if (lineHeight > 0) {
        const actualLines = Math.ceil(height / lineHeight)
        return actualLines
      }
    }
    
    // 回退到基于换行符的简单计算
    const lineBreaks = content.split('\n').length
    return lineBreaks
  }

  const shouldShowToggle = (timestamp: number): boolean => {
    const event = props.events.find(e => e.timestamp === timestamp)
    if (!event) return false
    
    const key = event.id || timestamp.toString()
    return (contentLineCounts.value[key] || 0) > 2
  }

  // 更新内容行数计数
  const updateLineCounts = () => {
    nextTick(() => {
      updateEventIdMapping()
      
      props.events.forEach((event) => {
        if ('content' in event && event.content) {
          const key = event.timestamp.toString()
          const element = contentElements.get(key)
          if (element) {
            updateLineCountForElement(event.timestamp)
          }
        }
      })
    })
  }

  return {
    getContentExpandedState,
    setContentRef,
    toggleContentExpand,
    shouldShowToggle,
    updateLineCounts
  }
}