package pangea.hiagent.tool.impl;

import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.tool.annotation.Tool;
import org.springframework.stereotype.Component;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.List;

/**
 * 存储文件访问工具类
 * 提供访问服务器后端 storage 目录下文件的功能，并将内容推送到对话界面的"网页预览区域"进行展示
 */
@Slf4j
@Component
public class StorageFileAccessTool {
    
    
    // 支持的文件扩展名
    private static final List<String> SUPPORTED_EXTENSIONS = Arrays.asList(
        ".txt", ".md", ".html", ".htm", ".xml", ".json", ".log", ".csv", 
        ".css", ".js"
    );
    
    // storage目录路径
    private static final String STORAGE_DIR = "backend" + File.separator + "storage";

    
    /**
     * 访问并预览storage目录下的文件
     * @param fileName 文件名（包含扩展名）
     * @return 操作结果描述
     */
    @Tool(description = "访问并预览storage目录下的文本类文件，支持.txt、.md、.html/.htm、.xml、.json、.log、.csv、.css、.js等格式")
    public String accessStorageFile(String fileName) {
        log.debug("接收到访问storage文件请求，文件名: {}", fileName);
        
        try {
            // 参数校验
            if (fileName == null || fileName.isEmpty()) {
                String result = "错误：文件名不能为空";
                log.warn(result);
                return result;
            }
            
            // 构建文件路径
            String filePath = STORAGE_DIR + File.separator + fileName;
            File file = new File(filePath);
            
            // 检查文件是否存在
            if (!file.exists()) {
                String result = "错误：文件不存在 - " + fileName;
                log.warn(result);
                return result;
            }
            
            // 检查是否为文件（而不是目录）
            if (!file.isFile()) {
                String result = "错误：指定路径不是一个文件 - " + fileName;
                log.warn(result);
                return result;
            }
            
            // 检查文件扩展名是否支持
            if (!isSupportedFile(fileName)) {
                String result = "错误：不支持的文件类型 - " + fileName;
                log.warn(result);
                return result;
            }
            
            // 读取文件内容
            String content = readFileContent(file);
            
            // 确定MIME类型
            String mimeType = getMimeType(fileName);
            String title = "文件预览: " + fileName;
            
            log.info("成功读取文件: {}", fileName);
            String result = "已成功在工作面板中预览文件: " + fileName;
        
            
            return result;
            
        } catch (Exception e) {
            log.error("访问文件时发生错误: {}, 错误详情: {}", fileName, e.getMessage(), e);
            return "访问文件时发生错误: " + e.getMessage();
        }
    }
    
    /**
     * 检查文件扩展名是否支持
     * @param fileName 文件名
     * @return 是否支持
     */
    private boolean isSupportedFile(String fileName) {
        String lowerFileName = fileName.toLowerCase();
        return SUPPORTED_EXTENSIONS.stream().anyMatch(lowerFileName::endsWith);
    }
    
    /**
     * 根据文件扩展名获取MIME类型
     * @param fileName 文件名
     * @return MIME类型
     */
    private String getMimeType(String fileName) {
        String lowerFileName = fileName.toLowerCase();
        if (lowerFileName.endsWith(".html") || lowerFileName.endsWith(".htm")) {
            return "text/html";
        } else if (lowerFileName.endsWith(".md")) {
            return "text/markdown";
        } else if (lowerFileName.endsWith(".xml")) {
            return "application/xml";
        } else if (lowerFileName.endsWith(".json")) {
            return "application/json";
        } else if (lowerFileName.endsWith(".css")) {
            return "text/css";
        } else if (lowerFileName.endsWith(".js")) {
            return "application/javascript";
        } else {
            return "text/plain";
        }
    }
    
    /**
     * 读取文件内容
     * @param file 文件对象
     * @return 文件内容
     * @throws Exception 读取异常
     */
    private String readFileContent(File file) throws Exception {
        StringBuilder content = new StringBuilder();
        BufferedReader reader = null;
        
        try {
            // 使用UTF-8编码读取文件
            reader = new BufferedReader(new InputStreamReader(new FileInputStream(file), StandardCharsets.UTF_8));
            String line;
            while ((line = reader.readLine()) != null) {
                content.append(line).append("\n");
            }
            
            log.debug("成功读取文件内容，长度: {} 字符", content.length());
            return content.toString();
        } finally {
            if (reader != null) {
                try {
                    reader.close();
                } catch (Exception e) {
                    log.debug("关闭读取器失败: {}", e.getMessage());
                }
            }
        }
    }
}