package pangea.hiagent.agent.data;

import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import pangea.hiagent.common.utils.ObjectPool;

import java.util.HashMap;
import java.util.Map;

/**
 * Map对象池服务
 * 提供可重用的HashMap实例，减少内存分配和垃圾回收压力
 */
@Slf4j
@Service
public class MapPoolService {
    
    // 使用通用对象池，用于重用HashMap实例，减少垃圾回收压力
    private final ObjectPool<Map<String, Object>> mapPool = new ObjectPool<Map<String, Object>>(
        () -> new HashMap<>(16), // 指定初始容量，避免频繁扩容
        200 // 对象池最大大小
    );
    
    /**
     * 从对象池获取HashMap实例
     */
    public Map<String, Object> acquireMap() {
        return mapPool.acquire();
    }
    
    /**
     * 将HashMap实例归还到对象池
     */
    public void releaseMap(Map<String, Object> map) {
        if (map != null) {
            map.clear(); // 清空复用的map
            mapPool.release(map);
        }
    }
    
    /**
     * 清空对象池（用于测试或清理）
     */
    public void clearMapPool() {
        mapPool.clear();
    }
    
    /**
     * 获取对象池当前大小
     */
    public int getMapPoolSize() {
        return mapPool.size();
    }
    
    /**
     * 获取对象池统计信息
     */
    public String getMapPoolStatistics() {
        return mapPool.getStatistics();
    }
}

