package pangea.hiagent.service;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import lombok.extern.slf4j.Slf4j;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import pangea.hiagent.model.PromptTemplate;
import pangea.hiagent.repository.PromptTemplateRepository;

import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 提示词模板服务类
 * 负责提示词模板的管理和渲染
 */
@Slf4j
@Service
public class PromptTemplateService {

    private final PromptTemplateRepository promptTemplateRepository;

    // 模板变量正则表达式：{{variableName}}
    private static final Pattern TEMPLATE_VARIABLE_PATTERN = Pattern.compile("\\{\\{(\\w+)\\}\\}");

    public PromptTemplateService(PromptTemplateRepository promptTemplateRepository) {
        this.promptTemplateRepository = promptTemplateRepository;
    }

    /**
     * 创建提示词模板
     */
    @Transactional
    @CacheEvict(value = {"promptTemplates", "promptTemplate"}, allEntries = true)
    public PromptTemplate createTemplate(PromptTemplate template) {
        log.info("创建提示词模板: {}", template.getName());

        // 设置默认值
        if (template.getIsSystem() == null) {
            template.setIsSystem(0); // 默认自定义模板
        }

        promptTemplateRepository.insert(template);
        return template;
    }

    /**
     * 更新提示词模板
     */
    @Transactional
    @CacheEvict(value = {"promptTemplates", "promptTemplate"}, allEntries = true)
    public PromptTemplate updateTemplate(PromptTemplate template) {
        log.info("更新提示词模板: {}", template.getId());

        // 获取现有模板
        PromptTemplate existingTemplate = promptTemplateRepository.selectById(template.getId());
        if (existingTemplate != null) {
            // 保留原始创建信息
            template.setCreatedBy(existingTemplate.getCreatedBy());
            template.setCreatedAt(existingTemplate.getCreatedAt());
            // 系统模板不允许修改isSystem属性
            template.setIsSystem(existingTemplate.getIsSystem());
        }

        promptTemplateRepository.updateById(template);
        return template;
    }

    /**
     * 删除提示词模板
     */
    @Transactional
    @CacheEvict(value = {"promptTemplates", "promptTemplate"}, allEntries = true)
    public void deleteTemplate(String id) {
        log.info("删除提示词模板: {}", id);
        promptTemplateRepository.deleteById(id);
    }

    /**
     * 获取提示词模板详情
     */
    @Cacheable(value = "promptTemplate", key = "#id")
    public PromptTemplate getTemplateById(String id) {
        if (id == null || id.isEmpty()) {
            log.warn("尝试使用无效ID获取提示词模板");
            return null;
        }
        return promptTemplateRepository.selectById(id);
    }

    /**
     * 获取提示词模板列表
     */
    @Cacheable(value = "promptTemplates")
    public List<PromptTemplate> listTemplates() {
        List<PromptTemplate> templates = promptTemplateRepository.selectList(null);
        log.info("获取到 {} 个提示词模板", templates != null ? templates.size() : 0);
        return templates != null ? templates : List.of();
    }

    /**
     * 根据类型获取提示词模板列表
     */
    @Cacheable(value = "promptTemplates", key = "#templateType")
    public List<PromptTemplate> listTemplatesByType(String templateType) {
        LambdaQueryWrapper<PromptTemplate> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(PromptTemplate::getTemplateType, templateType);
        return promptTemplateRepository.selectList(wrapper);
    }

    /**
     * 渲染提示词模板
     * 替换模板中的变量为实际值
     */
    public String renderTemplate(String templateContent, Map<String, Object> params) {
        if (templateContent == null || templateContent.isEmpty()) {
            return "";
        }

        log.debug("渲染提示词模板，参数: {}", params);
        String renderedContent = templateContent;
        Matcher matcher = TEMPLATE_VARIABLE_PATTERN.matcher(renderedContent);

        while (matcher.find()) {
            String variableName = matcher.group(1);
            String placeholder = matcher.group(0);
            Object value = params.get(variableName);

            if (value != null) {
                renderedContent = renderedContent.replace(placeholder, value.toString());
            } else {
                // 如果参数不存在，保留原始占位符
                log.warn("模板变量 {} 未提供值", variableName);
            }
        }

        log.debug("渲染后的提示词: {}", renderedContent);
        return renderedContent;
    }

    /**
     * 渲染提示词模板（根据模板ID）
     */
    public String renderTemplateById(String templateId, Map<String, Object> params) {
        PromptTemplate template = getTemplateById(templateId);
        if (template == null) {
            throw new IllegalArgumentException("提示词模板不存在: " + templateId);
        }
        return renderTemplate(template.getTemplateContent(), params);
    }

    /**
     * 验证提示词模板语法
     */
    public boolean validateTemplateSyntax(String templateContent) {
        if (templateContent == null || templateContent.isEmpty()) {
            return true;
        }

        // 简单验证：检查是否有未闭合的{{}}
        int openCount = 0;
        for (int i = 0; i < templateContent.length() - 1; i++) {
            if (templateContent.charAt(i) == '{' && templateContent.charAt(i + 1) == '{') {
                openCount++;
            } else if (templateContent.charAt(i) == '}' && templateContent.charAt(i + 1) == '}') {
                openCount--;
                if (openCount < 0) {
                    return false;
                }
            }
        }
        return openCount == 0;
    }
}
