package pangea.hiagent.common.config;

import com.baomidou.mybatisplus.core.handlers.MetaObjectHandler;
import lombok.extern.slf4j.Slf4j;
import org.apache.ibatis.reflection.MetaObject;
import org.springframework.stereotype.Component;
import pangea.hiagent.common.utils.UserUtils;

import java.time.LocalDateTime;

/**
 * MyBatis Plus自动填充配置类
 * 用于自动填充实体类中的公共字段
 */
@Slf4j
@Component
public class MetaObjectHandlerConfig implements MetaObjectHandler {

    /**
     * 插入时自动填充
     * @param metaObject 元对象
     */
    @Override
    public void insertFill(MetaObject metaObject) {
        log.debug("开始执行插入自动填充");
        
        // 自动填充创建时间
        if (metaObject.hasSetter("createdAt")) {
            Object createdAt = getFieldValByName("createdAt", metaObject);
            if (createdAt == null) {
                this.strictInsertFill(metaObject, "createdAt", LocalDateTime.class, LocalDateTime.now());
                log.debug("自动填充createdAt字段");
            }
        }
        
        // 自动填充更新时间
        if (metaObject.hasSetter("updatedAt")) {
            Object updatedAt = getFieldValByName("updatedAt", metaObject);
            if (updatedAt == null) {
                this.strictInsertFill(metaObject, "updatedAt", LocalDateTime.class, LocalDateTime.now());
                log.debug("自动填充updatedAt字段");
            }
        }
        
        // 自动填充创建人
        if (metaObject.hasSetter("createdBy")) {
            Object createdBy = getFieldValByName("createdBy", metaObject);
            if (createdBy == null) {
                String userId = UserUtils.getCurrentUserId();
                if (userId != null) {
                    this.strictInsertFill(metaObject, "createdBy", String.class, userId);
                    log.debug("自动填充createdBy字段: {}", userId);
                } else {
                    log.warn("无法获取当前用户ID，createdBy字段未填充");
                }
            }
        }
        
        // 自动填充更新人
        if (metaObject.hasSetter("updatedBy")) {
            Object updatedBy = getFieldValByName("updatedBy", metaObject);
            if (updatedBy == null) {
                String userId = UserUtils.getCurrentUserId();
                if (userId != null) {
                    this.strictInsertFill(metaObject, "updatedBy", String.class, userId);
                    log.debug("自动填充updatedBy字段: {}", userId);
                } else {
                    log.warn("无法获取当前用户ID，updatedBy字段未填充");
                }
            }
        }
    }

    /**
     * 更新时自动填充
     * @param metaObject 元对象
     */
    @Override
    public void updateFill(MetaObject metaObject) {
        log.debug("开始执行更新自动填充");
        
        // 自动填充更新时间
        if (metaObject.hasSetter("updatedAt")) {
            // 更新时总是设置updatedAt字段
            this.strictUpdateFill(metaObject, "updatedAt", LocalDateTime.class, LocalDateTime.now());
            log.debug("自动填充updatedAt字段");
        }
        
        // 自动填充更新人
        if (metaObject.hasSetter("updatedBy")) {
            Object updatedBy = getFieldValByName("updatedBy", metaObject);
            // 如果updatedBy为空或者需要强制更新，则填充当前用户ID
            if (updatedBy == null) {
                String userId = UserUtils.getCurrentUserId();
                if (userId != null) {
                    this.strictUpdateFill(metaObject, "updatedBy", String.class, userId);
                    log.debug("自动填充updatedBy字段: {}", userId);
                } else {
                    log.warn("无法获取当前用户ID，updatedBy字段未填充");
                }
            }
        }
    }
}