package pangea.hiagent.tool.impl;

import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.tool.annotation.Tool;
import org.springframework.stereotype.Component;
import pangea.hiagent.rag.RagService;
import pangea.hiagent.web.service.AgentService;
import pangea.hiagent.model.Agent;

import java.util.List;

/**
 * 课程资料检索工具
 * 用于检索和查询相关课程资料
 */
@Slf4j
@Component
public class CourseMaterialRetrievalTool {
    
    // RAG服务引用
    private RagService ragService;
    
    // Agent服务引用
    private AgentService agentService;

    public CourseMaterialRetrievalTool(RagService ragService, AgentService agentService) {
        this.ragService = ragService;
        this.agentService = agentService;
    }
    
    /**
     * 检索课程资料
     * @param query 查询关键词
     * @param subject 学科领域
     * @param maxResults 最大返回结果数
     * @return 检索到的课程资料内容
     */
    @Tool(description = "根据关键词和学科领域检索相关课程资料")
    public String searchCourseMaterials(String query, String subject, Integer maxResults) {
        log.debug("开始检索课程资料: 查询={}, 学科={}, 最大结果数={}", query, subject, maxResults);
        
        try {
            if (query == null || query.trim().isEmpty()) {
                log.warn("查询关键词不能为空");
                return "错误：查询关键词不能为空";
            }
            
            // 设置默认最大结果数
            if (maxResults == null || maxResults <= 0) {
                maxResults = 5;
            }
            
            // 获取学习导师Agent
            Agent learningMentorAgent = getLearningMentorAgent();
            if (learningMentorAgent == null) {
                log.error("未找到学习导师Agent");
                return "错误：未找到学习导师Agent配置";
            }
            
            // 使用RAG服务检索资料
            List<org.springframework.ai.document.Document> documents = 
                ragService.searchDocumentsByAgent(learningMentorAgent, query);
            
            // 限制返回结果数量
            if (documents.size() > maxResults) {
                documents = documents.subList(0, maxResults);
            }
            
            // 格式化结果
            StringBuilder result = new StringBuilder();
            result.append("课程资料检索结果：\n");
            result.append("查询关键词: ").append(query).append("\n");
            if (subject != null && !subject.trim().isEmpty()) {
                result.append("学科领域: ").append(subject).append("\n");
            }
            result.append("找到 ").append(documents.size()).append(" 个相关资料片段\n\n");
            
            for (int i = 0; i < documents.size(); i++) {
                org.springframework.ai.document.Document doc = documents.get(i);
                result.append("资料 ").append(i + 1).append(":\n");
                result.append(doc.getText()).append("\n\n");
            }
            
            log.info("课程资料检索完成，找到 {} 个结果", documents.size());
            return result.toString();
        } catch (Exception e) {
            log.error("检索课程资料时发生错误: {}", e.getMessage(), e);
            return "检索课程资料时发生错误: " + e.getMessage();
        }
    }
    
    /**
     * 获取学习导师Agent
     * @return 学习导师Agent对象
     */
    private Agent getLearningMentorAgent() {
        try {
            // 在实际应用中，这里应该通过某种方式获取学习导师Agent
            // 例如通过AgentService查询特定名称的Agent
            List<Agent> agents = agentService.listAgents();
            for (Agent agent : agents) {
                if ("学习导师".equals(agent.getName())) {
                    return agent;
                }
            }
        } catch (Exception e) {
            log.error("获取学习导师Agent时发生错误: {}", e.getMessage(), e);
        }
        return null;
    }
}