package pangea.hiagent.tool.impl;

import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.tool.annotation.Tool;
import org.springframework.stereotype.Component;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;

/**
 * 学习计划制定工具
 * 用于根据学习目标和时间安排制定个性化的学习计划
 */
@Slf4j
@Component
public class StudyPlanGenerationTool {
    
    public StudyPlanGenerationTool() {
        // 默认构造器
    }
    
    /**
     * 生成学习计划
     * @param subject 学科主题
     * @param goal 学习目标
     * @param startDate 开始日期
     * @param endDate 结束日期
     * @param hoursPerDay 每天学习小时数
     * @return 学习计划内容
     */
    @Tool(description = "根据学科主题、学习目标和时间安排生成个性化的学习计划")
    public String generateStudyPlan(String subject, String goal, String startDate, String endDate, Double hoursPerDay) {
        log.debug("开始生成学习计划: 学科={}, 目标={}, 开始日期={}, 结束日期={}, 每天学习小时数={}", 
                 subject, goal, startDate, endDate, hoursPerDay);
        
        try {
            if (subject == null || subject.trim().isEmpty()) {
                log.warn("学科主题不能为空");
                return "错误：学科主题不能为空";
            }
            
            if (goal == null || goal.trim().isEmpty()) {
                log.warn("学习目标不能为空");
                return "错误：学习目标不能为空";
            }
            
            // 验证日期格式
            LocalDate startLocalDate, endLocalDate;
            try {
                DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd");
                startLocalDate = LocalDate.parse(startDate, formatter);
                endLocalDate = LocalDate.parse(endDate, formatter);
            } catch (Exception e) {
                log.warn("日期格式不正确，应为yyyy-MM-dd格式");
                return "错误：日期格式不正确，应为yyyy-MM-dd格式";
            }
            
            // 验证时间范围
            if (endLocalDate.isBefore(startLocalDate)) {
                log.warn("结束日期不能早于开始日期");
                return "错误：结束日期不能早于开始日期";
            }
            
            // 设置默认学习时间
            if (hoursPerDay == null || hoursPerDay <= 0) {
                hoursPerDay = 2.0;
            }
            
            // 生成学习计划
            String studyPlan = generateStudyPlanContent(subject, goal, startLocalDate, endLocalDate, hoursPerDay);
            
            log.info("学习计划生成完成: 学科={}, 目标={}", subject, goal);
            return studyPlan;
        } catch (Exception e) {
            log.error("生成学习计划时发生错误: {}", e.getMessage(), e);
            return "生成学习计划时发生错误: " + e.getMessage();
        }
    }
    
    /**
     * 生成学习计划内容
     * @param subject 学科主题
     * @param goal 学习目标
     * @param startDate 开始日期
     * @param endDate 结束日期
     * @param hoursPerDay 每天学习小时数
     * @return 学习计划内容
     */
    private String generateStudyPlanContent(String subject, String goal, LocalDate startDate, LocalDate endDate, Double hoursPerDay) {
        StringBuilder plan = new StringBuilder();
        plan.append("个性化学习计划\n\n");
        plan.append("学科主题: ").append(subject).append("\n");
        plan.append("学习目标: ").append(goal).append("\n");
        plan.append("计划周期: ").append(startDate).append(" 至 ").append(endDate).append("\n");
        plan.append("每日学习时间: ").append(hoursPerDay).append("小时\n\n");
        
        // 计算总天数
        long totalDays = java.time.temporal.ChronoUnit.DAYS.between(startDate, endDate) + 1;
        plan.append("总学习天数: ").append(totalDays).append("天\n");
        plan.append("总学习时间: ").append(totalDays * hoursPerDay).append("小时\n\n");
        
        // 根据学科生成学习阶段
        plan.append("学习阶段安排:\n");
        
        switch (subject.toLowerCase()) {
            case "数学":
                plan.append("第一阶段 (第1-").append(Math.max(1, totalDays/3)).append("天):\n");
                plan.append("  - 基础概念复习\n");
                plan.append("  - 核心公式梳理\n");
                plan.append("  - 典型例题解析\n\n");
                
                plan.append("第二阶段 (第").append(Math.max(1, totalDays/3)+1).append("-").append(Math.max(1, 2*totalDays/3)).append("天):\n");
                plan.append("  - 专题训练\n");
                plan.append("  - 错题整理\n");
                plan.append("  - 知识点强化\n\n");
                
                plan.append("第三阶段 (第").append(Math.max(1, 2*totalDays/3)+1).append("-").append(totalDays).append("天):\n");
                plan.append("  - 综合练习\n");
                plan.append("  - 模拟测试\n");
                plan.append("  - 查漏补缺\n");
                break;
                
            case "英语":
                plan.append("第一阶段 (第1-").append(Math.max(1, totalDays/4)).append("天):\n");
                plan.append("  - 词汇积累\n");
                plan.append("  - 语法复习\n");
                plan.append("  - 听力训练\n\n");
                
                plan.append("第二阶段 (第").append(Math.max(1, totalDays/4)+1).append("-").append(Math.max(1, totalDays/2)).append("天):\n");
                plan.append("  - 阅读理解专项训练\n");
                plan.append("  - 写作技巧提升\n");
                plan.append("  - 口语练习\n\n");
                
                plan.append("第三阶段 (第").append(Math.max(1, totalDays/2)+1).append("-").append(Math.max(1, 3*totalDays/4)).append("天):\n");
                plan.append("  - 真题演练\n");
                plan.append("  - 错题分析\n");
                plan.append("  - 弱项突破\n\n");
                
                plan.append("第四阶段 (第").append(Math.max(1, 3*totalDays/4)+1).append("-").append(totalDays).append("天):\n");
                plan.append("  - 综合模拟\n");
                plan.append("  - 时间管理训练\n");
                plan.append("  - 心态调整\n");
                break;
                
            default:
                plan.append("第一阶段 (第1-").append(Math.max(1, totalDays/3)).append("天):\n");
                plan.append("  - 基础知识梳理\n");
                plan.append("  - 核心概念理解\n");
                plan.append("  - 典型案例学习\n\n");
                
                plan.append("第二阶段 (第").append(Math.max(1, totalDays/3)+1).append("-").append(Math.max(1, 2*totalDays/3)).append("天):\n");
                plan.append("  - 专题深入学习\n");
                plan.append("  - 实践应用训练\n");
                plan.append("  - 疑难问题解决\n\n");
                
                plan.append("第三阶段 (第").append(Math.max(1, 2*totalDays/3)+1).append("-").append(totalDays).append("天):\n");
                plan.append("  - 综合能力提升\n");
                plan.append("  - 模拟测试演练\n");
                plan.append("  - 总结反思优化\n");
                break;
        }
        
        plan.append("\n每日学习建议:\n");
        plan.append("- 制定每日具体学习任务\n");
        plan.append("- 记录学习进度和收获\n");
        plan.append("- 定期回顾和调整计划\n");
        plan.append("- 保持良好的作息习惯\n");
        
        plan.append("\n学习资源推荐:\n");
        plan.append("- 教材和参考书\n");
        plan.append("- 在线课程和视频\n");
        plan.append("- 练习题和模拟试卷\n");
        plan.append("- 学习小组和讨论社区\n");
        
        return plan.toString();
    }
}