package pangea.hiagent.web.service;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import pangea.hiagent.model.OAuth2Provider;
import pangea.hiagent.web.repository.OAuth2ProviderRepository;

import java.util.List;

/**
 * OAuth2 提供者服务类
 * 负责OAuth2提供商配置的管理和相关业务逻辑
 */
@Slf4j
@Service
public class OAuth2ProviderService {
    
    private final OAuth2ProviderRepository oAuth2ProviderRepository;
    
    public OAuth2ProviderService(OAuth2ProviderRepository oAuth2ProviderRepository) {
        this.oAuth2ProviderRepository = oAuth2ProviderRepository;
    }
    
    /**
     * 创建OAuth2提供商配置
     * 
     * @param provider OAuth2提供商配置对象
     * @return 创建后的OAuth2提供商配置对象
     */
    @Transactional
    public OAuth2Provider createProvider(OAuth2Provider provider) {
        log.info("创建OAuth2提供商配置: providerName={}", provider.getProviderName());
        oAuth2ProviderRepository.insert(provider);
        log.info("OAuth2提供商配置创建成功: id={}", provider.getId());
        return provider;
    }
    
    /**
     * 更新OAuth2提供商配置
     * 
     * @param id 提供商ID
     * @param provider 更新的OAuth2提供商配置对象
     * @return 更新后的OAuth2提供商配置对象
     */
    @Transactional
    public OAuth2Provider updateProvider(String id, OAuth2Provider provider) {
        log.info("更新OAuth2提供商配置: id={}", id);
        
        OAuth2Provider existingProvider = oAuth2ProviderRepository.selectById(id);
        if (existingProvider == null) {
            log.warn("尝试更新不存在的OAuth2提供商配置: id={}", id);
            throw new RuntimeException("OAuth2提供商配置不存在");
        }
        
        // 更新字段
        existingProvider.setProviderName(provider.getProviderName());
        existingProvider.setDisplayName(provider.getDisplayName());
        existingProvider.setDescription(provider.getDescription());
        existingProvider.setAuthType(provider.getAuthType());
        existingProvider.setAuthorizeUrl(provider.getAuthorizeUrl());
        existingProvider.setTokenUrl(provider.getTokenUrl());
        existingProvider.setUserinfoUrl(provider.getUserinfoUrl());
        existingProvider.setClientId(provider.getClientId());
        existingProvider.setClientSecret(provider.getClientSecret());
        existingProvider.setRedirectUri(provider.getRedirectUri());
        existingProvider.setScope(provider.getScope());
        existingProvider.setEnabled(provider.getEnabled());
        existingProvider.setConfigJson(provider.getConfigJson());
        existingProvider.setUpdatedBy(provider.getUpdatedBy());
        
        oAuth2ProviderRepository.updateById(existingProvider);
        log.info("OAuth2提供商配置更新成功: id={}", id);
        return existingProvider;
    }
    
    /**
     * 删除OAuth2提供商配置
     * 
     * @param id 提供商ID
     */
    @Transactional
    public void deleteProvider(String id) {
        log.info("删除OAuth2提供商配置: id={}", id);
        oAuth2ProviderRepository.deleteById(id);
        log.info("OAuth2提供商配置删除成功: id={}", id);
    }
    
    /**
     * 根据ID获取OAuth2提供商配置
     * 
     * @param id 提供商ID
     * @return OAuth2提供商配置对象，如果不存在则返回null
     */
    public OAuth2Provider getProviderById(String id) {
        if (id == null || id.isEmpty()) {
            log.warn("尝试使用无效ID获取OAuth2提供商配置");
            return null;
        }
        return oAuth2ProviderRepository.selectById(id);
    }
    
    /**
     * 根据提供商名称获取OAuth2提供商配置
     * 
     * @param providerName 提供商名称
     * @return OAuth2提供商配置对象，如果不存在则返回null
     */
    public OAuth2Provider getProviderByName(String providerName) {
        if (providerName == null || providerName.isEmpty()) {
            log.warn("尝试使用无效提供商名称获取OAuth2提供商配置");
            return null;
        }
        
        LambdaQueryWrapper<OAuth2Provider> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(OAuth2Provider::getProviderName, providerName);
        return oAuth2ProviderRepository.selectOne(wrapper);
    }
    
    /**
     * 获取OAuth2提供商配置列表
     * 
     * @return OAuth2提供商配置列表
     */
    public List<OAuth2Provider> listProviders() {
        List<OAuth2Provider> providers = oAuth2ProviderRepository.selectList(null);
        log.info("获取到 {} 个OAuth2提供商配置", providers != null ? providers.size() : 0);
        return providers != null ? providers : List.of();
    }
    
    /**
     * 分页获取OAuth2提供商配置列表
     * 
     * @param current 当前页码
     * @param size 每页大小
     * @param providerName 提供商名称（模糊查询）
     * @param displayName 显示名称（模糊查询）
     * @param enabled 启用状态
     * @return 分页结果
     */
    public IPage<OAuth2Provider> pageProviders(Long current, Long size, String providerName, String displayName, Integer enabled) {
        Page<OAuth2Provider> page = new Page<>(current, size);
        
        LambdaQueryWrapper<OAuth2Provider> wrapper = new LambdaQueryWrapper<>();
        if (providerName != null && !providerName.isEmpty()) {
            wrapper.like(OAuth2Provider::getProviderName, providerName);
        }
        if (displayName != null && !displayName.isEmpty()) {
            wrapper.like(OAuth2Provider::getDisplayName, displayName);
        }
        if (enabled != null) {
            wrapper.eq(OAuth2Provider::getEnabled, enabled);
        }
        
        return oAuth2ProviderRepository.selectPage(page, wrapper);
    }
    
    /**
     * 启用OAuth2提供商配置
     * 
     * @param id 提供商ID
     */
    @Transactional
    public void enableProvider(String id) {
        log.info("启用OAuth2提供商配置: id={}", id);
        OAuth2Provider provider = oAuth2ProviderRepository.selectById(id);
        if (provider != null) {
            provider.setEnabled(1);
            oAuth2ProviderRepository.updateById(provider);
            log.info("OAuth2提供商配置启用成功: id={}", id);
        } else {
            log.warn("尝试启用不存在的OAuth2提供商配置: id={}", id);
            throw new RuntimeException("OAuth2提供商配置不存在");
        }
    }
    
    /**
     * 禁用OAuth2提供商配置
     * 
     * @param id 提供商ID
     */
    @Transactional
    public void disableProvider(String id) {
        log.info("禁用OAuth2提供商配置: id={}", id);
        OAuth2Provider provider = oAuth2ProviderRepository.selectById(id);
        if (provider != null) {
            provider.setEnabled(0);
            oAuth2ProviderRepository.updateById(provider);
            log.info("OAuth2提供商配置禁用成功: id={}", id);
        } else {
            log.warn("尝试禁用不存在的OAuth2提供商配置: id={}", id);
            throw new RuntimeException("OAuth2提供商配置不存在");
        }
    }
}