package pangea.hiagent.agent.react;

import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.chat.client.ChatClient;
import org.springframework.ai.chat.messages.*;
import org.springframework.ai.chat.model.ChatResponse;
import org.springframework.ai.chat.prompt.Prompt;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import pangea.hiagent.agent.service.ErrorHandlerService;
import pangea.hiagent.agent.service.TokenConsumerWithCompletion;
import pangea.hiagent.memory.MemoryService;
import pangea.hiagent.model.Agent;
import pangea.hiagent.tool.AgentToolManager;
import pangea.hiagent.tool.impl.DateTimeTools;
import pangea.hiagent.common.utils.UserUtils;
import java.util.List;
import java.util.ArrayList;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;

/**
 * 简化的默认ReAct执行器实现
 */
@Slf4j
@Service
public class DefaultReactExecutor implements ReactExecutor {
    
    private static final String DEFAULT_SYSTEM_PROMPT = 
        "You are a powerful professional AI assistant powered by the enhanced ReAct (Reasoning + Acting) iterative framework, specialized for Spring AI tool orchestration. Your core mission is to solve complex, multi-step user queries with high accuracy by following the upgraded rules. The TOP PRIORITY principle is: ALWAYS CALL TOOLS FIRST, and answer questions EXCLUSIVELY based on tool execution results. You have full authority to intelligently select, combine, and serially invoke multiple tools, and iterate reasoning until a complete and satisfactory answer is obtained.\n\n" +
        "=== CORE UPGRADED RULE - NON-NEGOTIABLE (Tool-First Priority Highlighted) ===\n\n" +
        "1. Tool-First Mandate: For any query that requires factual verification, data calculation, information extraction, content analysis, or scenario-based processing, YOU MUST CALL RELEVANT TOOLS FIRST. Never answer directly relying on internal knowledge without tool invocation, except for extremely simple common-sense questions (e.g., \"What is 1+1?\").\n" +
        "2. Answer Based on Tool Results Only: All conclusions, data, and insights in the final answer must be strictly derived from the real execution results of Spring AI tools. Never fabricate any data, assumptions, or inferences that are not supported by tool outputs.\n" +
        "3. Serial Multi-Tool Invocation Supported: You can invoke multiple tools in serial order in one Action phase. By default, the output of the previous tool is the directly valid input of the next tool (first-class support for tool chaining).\n" +
        "4. Iterative ReAct Closed-Loop: The ReAct thinking process is a cyclic loop. After each Observation phase, you can return to the Thought phase to re-analyze, reselect tools, and re-execute until the answer is complete/satisfactory.\n" +
        "5. Mandatory Tool Synergy: Complex queries must use multi-tool combinations. A single tool can only solve simple problems; never rely on a single tool for complex tasks.\n" +
        "6. Strict Compliance with Spring AI Mechanism: All tool calls are executed automatically by the Spring AI framework. You only need to make optimal tool selection and sequence planning.\n\n\n" +
        "=== ENHANCED TOOL SYNERGY & ORCHESTRATION STRATEGY ===\n\n" +
        "You have access to a full set of specialized Spring AI tools and must create value through intelligent tool collocation, with tool-first logic throughout:\n\n" +
        "- Serial Chaining (Highest Priority): The output of one tool directly feeds into the input of another, forming a closed tool call chain (e.g., File Reader → Text Processor → Calculator → File Writer → Chart Generator).\n\n" +
        "- Parallel Combination: Call multiple independent tools simultaneously to collect multi-dimensional data, then merge results for comprehensive analysis.\n\n" +
        "- Preprocessing & Postprocessing: Use formatting tools to clean raw data before core tool execution; use conversion tools to optimize result presentation afterward.\n\n" +
        "- Layered Enrichment: Combine extraction, analysis, and calculation tools to gain in-depth insights instead of superficial data.\n\n" +
        "- Priority Matching: Select lightweight tools first for simple sub-tasks; use heavyweight tools only for complex ones (resource efficiency).\n\n" +
        "- Fault Tolerance Fallback: If a selected tool is unavailable/returns invalid results, immediately invoke an alternative tool with the same function to re-execute the sub-task.\n\n\n" +
        "=== Typical High-Value Tool Synergy Examples ===\n\n" +
        "1. Web Content Extractor → Text Parser & Cleaner → NLP Analyzer → Statistical Calculator → Result Formatter → File Saver\n\n" +
        "2. Current DateTime Tool → Date Formatter → Data Filter → Time Series Analyzer → Visualization Tool\n\n" +
        "3. Document Reader → Table Extractor → Data Validator → Formula Calculator → Report Generator\n\n" +
        "4. Input Parameter Parser → Multiple Business Tools (Serial) → Result Aggregator → Answer Polisher\n\n\n" +
        "=== UPGRADED ITERATIVE ReAct THINKING PROCESS (Tool-First Oriented) ===\n\n" +
        "This is a cyclic, repeatable process for EVERY query, with tool-first logic as the core. Execute in order and loop infinitely until the answer meets completeness requirements.\n\n" +
        "▶ Cycle Trigger Rule: After Step 4 (Observation), if results are incomplete/insufficient/need optimization → Return to Step 1 (Thought) to re-analyze and re-execute.\n\n" +
        "▶ Cycle Termination Rule: After Step 4 (Observation), if results are complete/accurate/satisfactory → Enter Step 5 (Final Answer) directly.\n\n\n" +
        "Step 1 - THOUGHT (Tool-First Iterative Reasoning & Planning): Deeply analyze the user's core query and current context with tool-first logic\n" +
        "  - Break down the main problem into hierarchical sub-tasks (primary → secondary → fine-grained).\n" +
        "  - Tool-First Matching: For each sub-task, FIRST identify relevant tools (never consider direct answering first). Mark alternative tools for fault tolerance.\n" +
        "  - Confirm Tool Synergy Feasibility: Judge serial/parallel combination of multi-tools and define the exact invocation sequence.\n" +
        "  - Iterative Scenario Adjustment: Re-analyze the gap between current tool results and expected answers, adjust tool selection/sequence.\n" +
        "  - Verify Preconditions: Ensure input format and parameter validity for tool invocation are met.\n\n\n" +
        "Step 2 - ACTION (Multi-Tool Serial/Parallel Execution): Execute the planned tool chain with clear purpose, adhering to tool-first principle\n" +
        "  - Call tools in the pre-defined serial/parallel order based on Thought phase analysis.\n" +
        "  - Support multiple consecutive tool calls in one Action phase (serial chain) for Spring AI, no limit on the number of tools.\n" +
        "  - Wait for ALL tool execution results (serial: one by one / parallel: all at once) before proceeding; never jump early.\n" +
        "  - Fault Tolerance Execution: If a tool returns invalid/empty results, immediately invoke the pre-marked alternative tool and re-execute the sub-task.\n\n\n" +
        "Step 3 - OBSERVATION (Tool Result-Centric Analysis & Validation): Comprehensively interpret all tool execution results\n" +
        "  - Examine data/results from each tool in detail, cross-verify accuracy, completeness, and logical consistency.\n" +
        "  - Extract key information, patterns, and insights EXCLUSIVELY from combined tool results.\n" +
        "  - Judge Completion Status: Confirm if current results cover all sub-tasks and meet the user's core needs.\n" +
        "  - Identify Gaps: Mark missing information/unsolved sub-tasks that require further tool invocation.\n" +
        "  - Evaluate Tool Synergy Effect: Confirm if the tool chain provides deeper insights than single-tool usage.\n\n\n" +
        "Step 4 - ITERATION DECISION: Critical judgment for ReAct cycle\n" +
        "  - ✅ TERMINATE CYCLE: If observation results are complete, accurate, sufficient, and fully meet the user's query → Proceed to Step 5.\n\n" +
        "  ♻️ RESTART CYCLE: If observation results are incomplete/insufficient/have missing information → Return to Step 1.\n\n\n" +
        "Step 5 - FINAL ANSWER (Tool Result-Synthesized Response): Generate the ultimate answer based solely on tool results\n" +
        "  - Synthesize all valid tool results (from iterative cycles) into a coherent, logical, and complete answer.\n" +
        "  - Present information in clear, easy-to-understand natural language, distinguishing key insights from basic information.\n" +
        "  - Explicitly explain tool synergy logic (e.g., \"Tool A processed raw data for Tool B, enabling accurate calculation by Tool C\").\n" +
        "  - Provide actionable conclusions, recommendations, or follow-up suggestions based on integrated tool results.\n" +
        "  - Keep the answer conversational and business-oriented; remove redundant technical tool details.\n\n\n" +
        "=== STANDARDIZED RESPONSE FORMAT ===\n\n" +
        "Strictly follow this fixed structure for all responses to ensure correct parsing by Spring AI:\n\n\n" +
        "1. Thought: Detailed explanation of problem analysis, sub-task breakdown, tool-first selection strategy, and invocation sequence\n" +
        "   - Identified Sub-Problems: List all primary/secondary sub-tasks clearly.\n" +
        "   - Tool-First Matching: Tools assigned to each sub-task + alternative tools (if any).\n" +
        "   - Execution Sequence: Exact serial/parallel order of multi-tool invocation and its optimality.\n" +
        "   - Iteration Note: If re-analyzing (loop), explain gaps in previous results and tool selection adjustments.\n\n\n" +
        "2. Action: Clear description of all tool calls in this phase (serial number + tool name + core purpose)\n" +
        "   - Tool_Call: 1.[Tool Name] → Purpose: [Exact business objective and core value]\n" +
        "   - Tool_Call: 2.[Tool Name] → Purpose: [Complement the previous tool, use its output as input]\n" +
        "   - Tool_Call: N.[Tool Name] → Purpose: [Final enrichment/validation/formatting of the result chain]\n" +
        "   - (Fallback) If Tool X Unavailable: Use [Alternative Tool Name] → Purpose: [Same objective as Tool X]\n\n\n" +
        "3. Observation: Comprehensive interpretation of all tool execution results\n" +
        "   - Results from each individual tool (key data, no redundant details).\n" +
        "   - Logical relationship between multiple tool results (how they connect and complement).\n" +
        "   - Core patterns/insights from the tool chain.\n" +
        "   - Completion Status: Whether results cover all sub-tasks and missing information (if any).\n\n\n" +
        "4. Iteration_Decision: Explicit single choice\n" +
        "   - Option 1: Terminate Cycle → Proceed to Final Answer (complete results)\n" +
        "   - Option 2: Restart Cycle → Re-enter Thought phase (incomplete results)\n\n\n" +
        "5. Final_Answer: Polished, complete, and user-friendly natural language solution\n" +
        "   - Direct answer to the original query, with core conclusions first.\n" +
        "   - Highlight key insights from tool synergy/iterative reasoning.\n" +
        "   - Provide actionable follow-up suggestions.\n" +
        "   - Conversational tone; no technical jargon about tools/frameworks.\n\n\n\n" +
        "=== CRITICAL HARD RULES (Tool-First as Core) ===\n\n" +
        "1. Tool-First is Non-Negotiable: For non-trivial queries, call tools first. Never answer directly with internal knowledge unless it's extremely simple common sense.\n" +
        "2. Tool Results are the Sole Basis: All answers must rely on real Spring AI tool execution results. Never fabricate data/results.\n" +
        "3. Mandatory Multi-Tool Synergy: Complex queries must use tool combinations. Never rely on a single tool for complex tasks.\n" +
        "4. Full Support for Serial Invocation: One Action phase can call N tools in sequence, with prior output as next input.\n" +
        "5. Iterative ReAct is Mandatory: Never stop at one-time execution; loop until the answer is complete and satisfactory.\n" +
        "6. Explicit Tool Strategy: All tool selection, sequence planning, and fallback options must be clearly stated in Thought.\n" +
        "7. Unavailable Tool Handling: Immediately use an alternative tool if the selected one is unavailable; do not suspend execution.\n" +
        "8. User Experience Priority: The Final Answer must be conversational and business-focused, hiding technical tool details.\n" +
        "9. Spring AI Compliance: All tool calls follow the framework's automatic execution rules; no custom execution logic.";
    
    
    private final List<ReactCallback> reactCallbacks = new ArrayList<>();
    private final AtomicInteger stepCounter = new AtomicInteger(0);
    
    @Autowired
    private DateTimeTools dateTimeTools;
    
    @Autowired
    private MemoryService memoryService;
    
    @Autowired
    private ErrorHandlerService errorHandlerService;
    
    private final AgentToolManager agentToolManager;
    
    public DefaultReactExecutor(AgentToolManager agentToolManager) {
        this.agentToolManager = agentToolManager;
    }
    
    @Override
    public void addReactCallback(ReactCallback callback) {
        if (callback != null) {
            reactCallbacks.add(callback);
        }
    }
    
    @Override
    public String execute(ChatClient chatClient, String userInput, List<Object> tools, Agent agent) {
        // 调用带用户ID的方法，首先尝试获取当前用户ID
        String userId = UserUtils.getCurrentUserId();
        return execute(chatClient, userInput, tools, agent, userId);
    }
    
    @Override
    public String execute(ChatClient chatClient, String userInput, List<Object> tools, Agent agent, String userId) {
        log.info("开始执行ReAct流程，用户输入: {}", userInput);
        
        stepCounter.set(0);
        
        List<Object> agentTools = getAgentTools(agent);
        
        try {
            // triggerThinkStep("开始处理用户请求: " + userInput);
            
            Prompt prompt = buildPromptWithHistory(DEFAULT_SYSTEM_PROMPT, userInput, agent, userId);
            
            ChatResponse response = chatClient.prompt(prompt)
                    .tools(agentTools.toArray())
                    .call()
                    .chatResponse();
            
            String responseText = response.getResult().getOutput().getText();
            
            // triggerObservationStep(responseText);
            
            log.info("最终答案: {}", responseText);
            
            // triggerFinalAnswerStep(responseText);
            
            // 保存助手回复到内存，使用提供的用户ID
            saveAssistantResponseToMemory(agent, responseText, userId);
            
            return responseText;
        } catch (Exception e) {
            log.error("执行ReAct流程时发生错误", e);
            return handleReActError(e);
        }
    }
    
    /**
     * 处理ReAct执行过程中发生的错误
     * 
     * @param e 发生的异常
     * @return 错误处理结果
     */
    private String handleReActError(Exception e) {
        return errorHandlerService.handleSyncError(e, "处理ReAct请求时发生错误");
    }
    
    /**
     * 构建带有历史记录的提示词
     * 
     * @param systemPrompt 系统提示词
     * @param userInput 用户输入
     * @param agent 智能体对象
     * @param userId 用户ID（可选，如果为null则自动获取）
     * @return 构建好的提示词对象
     */
    private Prompt buildPromptWithHistory(String systemPrompt, String userInput, Agent agent, String userId) {
        List<org.springframework.ai.chat.messages.Message> messages = new ArrayList<>();
        
        messages.add(new SystemMessage(systemPrompt));
        
        if (agent != null) {
            try {
                // 如果没有提供用户ID，则尝试获取当前用户ID
                if (userId == null) {
                    userId = UserUtils.getCurrentUserId();
                }
                String sessionId = memoryService.generateSessionId(agent, userId);
                
                int historyLength = agent.getHistoryLength() != null ? agent.getHistoryLength() : 10;
                
                List<org.springframework.ai.chat.messages.Message> historyMessages = 
                    memoryService.getHistoryMessages(sessionId, historyLength);
                
                messages.addAll(historyMessages);
                
                memoryService.addUserMessageToMemory(sessionId, userInput);
            } catch (Exception e) {
                log.warn("获取历史对话记录时发生错误: {}", e.getMessage());
            }
        }
        
        messages.add(new UserMessage(userInput));
        
        return new Prompt(messages);
    }
    
    @Override
    public void executeStream(ChatClient chatClient, String userInput, List<Object> tools, Consumer<String> tokenConsumer, Agent agent) {
        // 调用带用户ID的方法，但首先尝试获取当前用户ID
        String userId = UserUtils.getCurrentUserId();
        executeStream(chatClient, userInput, tools, tokenConsumer, agent, userId);
    }
    
    @Override
    public void executeStream(ChatClient chatClient, String userInput, List<Object> tools, Consumer<String> tokenConsumer, Agent agent, String userId) {
        log.info("使用stream()方法处理ReAct流程，支持真正的流式输出");
        
        stepCounter.set(0);
        
        List<Object> agentTools = getAgentTools(agent);
        
        StringBuilder fullResponse = new StringBuilder();
        
        try {
            // triggerThinkStep("开始处理用户请求: " + userInput);
            
            Prompt prompt = buildPromptWithHistory(DEFAULT_SYSTEM_PROMPT, userInput, agent, userId);
            
            chatClient.prompt(prompt)
                    .tools(agentTools.toArray())
                    .stream()
                    .chatResponse()
                    .subscribe(
                        chatResponse -> handleTokenResponse(chatResponse, tokenConsumer, fullResponse),
                        throwable -> handleStreamError(throwable, tokenConsumer),
                        () -> handleStreamCompletion(tokenConsumer, fullResponse, agent, userId)
                    );
            
        } catch (Exception e) { 
            log.error("流式执行ReAct流程时发生错误", e);
            errorHandlerService.handleReactFlowError(e, tokenConsumer);
        }
    }
    
    /**
     * 处理流式响应中的单个token
     * 
     * @param chatResponse 聊天响应对象
     * @param tokenConsumer token消费者
     * @param fullResponse 完整响应构建器
     */
    private void handleTokenResponse(org.springframework.ai.chat.model.ChatResponse chatResponse, Consumer<String> tokenConsumer, StringBuilder fullResponse) {
        try {
            String token = chatResponse.getResult().getOutput().getText();
            
            if (isValidToken(token)) {
                fullResponse.append(token);
                
                // analyzeAndRecordToolEvents(token, fullResponse.toString());
                
                if (tokenConsumer != null) {
                    tokenConsumer.accept(token);
                }

                // tokenTextSegmenter.inputChar(token);
                // tokenTextSegmenter.finishInput();
                
                // 改进：在流式处理过程中实时解析关键词
                // processTokenForStepsWithFullResponse(token, fullResponse.toString());
            }
        } catch (Exception e) {
            log.error("处理token时发生错误", e);
        }
    }
    
    /**
     * 处理流式响应完成事件
     * 
     * @param tokenConsumer token消费者
     * @param fullResponse 完整响应内容
     * @param agent 智能体对象
     * @param userId 用户ID
     */
    private void handleStreamCompletion(Consumer<String> tokenConsumer, StringBuilder fullResponse, Agent agent, String userId) {
        try {
            log.info("流式处理完成");
            
            // 检查是否已经处理了Final Answer，如果没有，则将整个响应作为最终答案
            String responseStr = fullResponse.toString();
            if (!hasFinalAnswerBeenTriggered(responseStr)) {
                // triggerFinalAnswerStep(responseStr);
            }
            
            saveAssistantResponseToMemory(agent, responseStr, userId);
            
            sendCompletionEvent(tokenConsumer, responseStr);
        } catch (Exception e) {
            log.error("处理流式完成回调时发生错误", e);
            handleCompletionError(tokenConsumer, e);
        }
    }
    
    /**
     * 检查是否已经触发了Final Answer步骤
     * 
     * @param fullResponse 完整响应内容
     * @return 如果已经触发了Final Answer则返回true，否则返回false
     */
    private boolean hasFinalAnswerBeenTriggered(String fullResponse) {
        String[] finalAnswerPatterns = {"Final Answer:", "final answer:", "FINAL ANSWER:", "Final_Answer:", "final_answer:", "FINAL_ANSWER:", "最终答案:"};
        for (String pattern : finalAnswerPatterns) {
            if (fullResponse.toLowerCase().contains(pattern.toLowerCase())) {
                return true;
            }
        }
        return false;
    }
    
    /**
     * 将助手的回复保存到内存中
     * 
     * @param agent 智能体对象
     * @param response 助手的回复内容
     * @param userId 用户ID
     */
    private void saveAssistantResponseToMemory(Agent agent, String response, String userId) {
        if (agent != null) {
            try {
                String sessionId = memoryService.generateSessionId(agent, userId);
                memoryService.addAssistantMessageToMemory(sessionId, response);
            } catch (Exception e) {
                log.warn("保存助理回复到内存时发生错误: {}", e.getMessage());
            }
        }
    }
    
    /**
     * 处理完成事件时发生的错误
     * 
     * @param tokenConsumer token消费者
     * @param e 发生的异常
     */
    private void handleCompletionError(Consumer<String> tokenConsumer, Exception e) {
        if (tokenConsumer instanceof TokenConsumerWithCompletion) {
            try {
                String errorId = errorHandlerService.generateErrorId();
                String fullErrorMessage = errorHandlerService.buildFullErrorMessage("处理完成时发生错误", e, errorId, "ReAct");
                try {
                    ((TokenConsumerWithCompletion) tokenConsumer).onComplete("[" + errorId + "] " + fullErrorMessage);
                } catch (NoClassDefFoundError ex) {
                    log.error("TokenConsumerWithCompletion依赖类未找到，跳过完成回调: {}", ex.getMessage());
                }
            } catch (Exception ex) {
                log.error("调用onComplete时发生错误", ex);
            }
        }
    }
    
    /**
     * 验证token是否有效
     * 
     * @param token 待验证的token
     * @return 如果token有效则返回true，否则返回false
     */
    private boolean isValidToken(String token) {
        return token != null && !token.isEmpty();
    }
    
    /**
     * 处理流式响应中的错误
     * 
     * @param throwable 异常对象
     * @param tokenConsumer token消费者
     */
    private void handleStreamError(Throwable throwable, Consumer<String> tokenConsumer) {
        errorHandlerService.handleStreamError(throwable, tokenConsumer, "ReAct流式处理");
    }
    
    /**
     * 发送完成事件
     * 
     * @param tokenConsumer token消费者
     * @param fullResponse 完整响应内容
     */
    private void sendCompletionEvent(Consumer<String> tokenConsumer, String fullResponse) {
        if (fullResponse == null) {
            fullResponse = "";
        }
        
        if (tokenConsumer instanceof TokenConsumerWithCompletion) {
            try {
                ((TokenConsumerWithCompletion) tokenConsumer).onComplete(fullResponse);
            } catch (NoClassDefFoundError e) {
                log.error("TokenConsumerWithCompletion依赖类未找到，跳过完成回调: {}", e.getMessage());
                // 如果类未找到，至少发送一个空消息以确保流的完整性
                if (tokenConsumer != null) {
                    try {
                        tokenConsumer.accept("");
                    } catch (Exception ex) {
                        log.error("发送空消息也失败", ex);
                    }
                }
            } catch (Exception e) {
                log.error("调用onComplete时发生错误", e);
            }
        } else if (tokenConsumer != null) {
            tokenConsumer.accept("");
        }
    }
    
    /**
     * 获取智能体工具
     * 
     * @param agent 智能体对象
     * @return 智能体可用的工具列表
     */
    private List<Object> getAgentTools(Agent agent) {
        if (agent == null) {
            List<Object> defaultTools = new ArrayList<>();
            defaultTools.add(dateTimeTools);
            return defaultTools;
        }
        
        try {
            List<Object> tools = agentToolManager.getAvailableToolInstances(agent);
            
            if (dateTimeTools != null && !tools.contains(dateTimeTools)) {
                tools.add(dateTimeTools);
            }
            
            return tools;
        } catch (Exception e) {
            log.error("获取工具实例时发生错误: {}", e.getMessage());
            List<Object> fallbackTools = new ArrayList<>();
            fallbackTools.add(dateTimeTools);
            return fallbackTools;
        }
    }
}