package pangea.hiagent.common.utils;

import lombok.extern.slf4j.Slf4j;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Component;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;
import org.springframework.util.StringUtils;

import jakarta.servlet.http.HttpServletRequest;

/**
 * 用户相关工具类
 * 提供统一的用户信息获取方法
 */
@Slf4j
@Component
public class UserUtils {

    // 注入JwtUtil bean
    private static JwtUtil jwtUtil;

    public UserUtils(JwtUtil jwtUtil) {
        UserUtils.jwtUtil = jwtUtil;
    }

    public static String getCurrentUserId() {

        String username = getCurrentUserIdInSync();
        if (username==null || username.isEmpty()) {
            username = getCurrentUserIdInAsync();
        }

        return username;
    }

    /**
     * 获取当前认证用户ID
     * 
     * @return 用户ID，如果未认证则返回null
     */
    public static String getCurrentUserIdInSync() {
        try {
            // 首先尝试从SecurityContext获取
            Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
            if (authentication != null && authentication.isAuthenticated() && authentication.getPrincipal() != null) {
                Object principal = authentication.getPrincipal();
                if (principal instanceof String) {
                    String userId = (String) principal;
                    log.debug("从SecurityContext获取到用户ID: {}", userId);
                    return userId;
                } else {
                    // 如果principal不是String类型，尝试获取getName()方法的返回值
                    log.debug("Authentication principal is not a String: {}", principal.getClass().getName());
                    try {
                        String userId = principal.toString();
                        log.debug("将principal转换为字符串获取用户ID: {}", userId);
                        return userId;
                    } catch (Exception toStringEx) {
                        log.warn("无法将principal转换为字符串: {}", toStringEx.getMessage());
                    }
                }
            }

            // 如果SecurityContext中没有认证信息，尝试从请求中解析JWT令牌
            String userId = getUserIdFromRequest();
            if (userId != null) {
                log.debug("从请求中解析到用户ID: {}", userId);
                return userId;
            }

            log.debug("未能获取到有效的用户ID");
            return null;
        } catch (Exception e) {
            log.error("获取当前用户ID时发生异常", e);
            return null;
        }
    }

    /**
     * 在异步线程环境中获取当前认证用户ID
     * 该方法专为异步线程环境设计，通过JWT令牌解析获取用户ID
     * 
     * @return 用户ID，如果未认证则返回null
     */
    public static String getCurrentUserIdInAsync() {
        try {
            log.debug("在异步线程中尝试获取用户ID");

            // 直接从请求中解析JWT令牌获取用户ID
            String userId = getUserIdFromRequest();
            if (userId != null) {
                log.debug("在异步线程中成功获取用户ID: {}", userId);
                return userId;
            }

            log.debug("在异步线程中未能获取到有效的用户ID");
            return null;
        } catch (Exception e) {
            log.error("在异步线程中获取用户ID时发生异常", e);
            return null;
        }
    }

    /**
     * 从当前请求中提取JWT令牌并解析用户ID
     * 
     * @return 用户ID，如果无法解析则返回null
     */
    private static String getUserIdFromRequest() {
        try {
            RequestAttributes requestAttributes = RequestContextHolder.getRequestAttributes();
            if (requestAttributes instanceof ServletRequestAttributes) {
                HttpServletRequest request = ((ServletRequestAttributes) requestAttributes).getRequest();

                // 从请求头或参数中提取Token
                String token = extractTokenFromRequest(request);

                if (StringUtils.hasText(token) && jwtUtil != null) {
                    // 验证token是否有效
                    boolean isValid = jwtUtil.validateToken(token);
                    log.debug("JWT验证结果: {}", isValid);

                    if (isValid) {
                        String userId = jwtUtil.getUserIdFromToken(token);
                        log.debug("从JWT令牌中提取用户ID: {}", userId);
                        return userId;
                    } else {
                        log.warn("JWT验证失败，token可能已过期或无效");
                    }
                } else {
                    if (jwtUtil == null) {
                        log.error("jwtUtil未初始化");
                    } else {
                        log.debug("未找到有效的token");
                    }
                }
            } else {
                log.debug("无法获取请求上下文，可能在异步线程中调用");
            }
        } catch (Exception e) {
            log.error("从请求中解析用户ID时发生异常", e);
        }

        return null;
    }

    /**
     * 从请求头或参数中提取Token
     */
    private static String extractTokenFromRequest(HttpServletRequest request) {
        // 首先尝试从请求头中提取Token
        String authHeader = request.getHeader("Authorization");
        log.debug("从请求头中提取Authorization: {}", authHeader);
        if (StringUtils.hasText(authHeader) && authHeader.startsWith("Bearer ")) {
            String token = authHeader.substring(7);
            log.debug("从Authorization头中提取到token");
            return token;
        }

        // 如果请求头中没有Token，则尝试从URL参数中提取
        String tokenParam = request.getParameter("token");
        log.debug("从URL参数中提取token参数: {}", tokenParam);
        if (StringUtils.hasText(tokenParam)) {
            log.debug("从URL参数中提取到token");
            return tokenParam;
        }

        log.debug("未找到有效的token");
        return null;
    }

    /**
     * 检查当前用户是否已认证
     * 
     * @return true表示已认证，false表示未认证
     */
    public static boolean isAuthenticated() {
        return getCurrentUserId() != null;
    }

    /**
     * 检查用户是否是管理员
     * 
     * @param userId 用户ID
     * @return true表示是管理员，false表示不是管理员
     */
    public static boolean isAdminUser(String userId) {
        // 这里可以根据实际需求实现管理员检查逻辑
        // 例如查询数据库或检查特殊用户ID
        // 当前实现保留原有逻辑，但可以通过配置或数据库来管理管理员用户
        return "admin".equals(userId) || "user-001".equals(userId);
    }
}