package pangea.hiagent.agent.processor;

import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.chat.model.ChatModel;
import org.springframework.ai.chat.model.StreamingChatModel;
import org.springframework.ai.chat.prompt.Prompt;
import org.springframework.beans.factory.annotation.Autowired;
import pangea.hiagent.memory.MemoryService;
import pangea.hiagent.memory.SmartHistorySummarizer;
import pangea.hiagent.model.Agent;
import pangea.hiagent.rag.RagService;
import pangea.hiagent.agent.service.AgentErrorHandler;
import pangea.hiagent.web.service.AgentService;
import pangea.hiagent.agent.service.ErrorHandlerService;
import pangea.hiagent.agent.service.TokenConsumerWithCompletion;
import java.util.ArrayList;
import java.util.List;
import java.util.function.Consumer;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * Agent处理器抽象基类
 * 封装NormalAgent和ReActAgent的公共逻辑
 * 职责：提供所有Agent处理器共享的基础功能
 */
@Slf4j
public abstract class BaseAgentProcessor implements AgentProcessor {
    
    @Autowired
    protected AgentService agentService;
    
    @Autowired
    protected MemoryService memoryService;
    
    @Autowired
    protected SmartHistorySummarizer smartHistorySummarizer;
    
    @Autowired
    protected ErrorHandlerService errorHandlerService;
    
    @Autowired
    protected AgentErrorHandler agentErrorHandler;
    
    // 默认系统提示词
    protected static final String DEFAULT_SYSTEM_PROMPT = "你是一个智能助手";
    
    /**
     * 生成会话ID
     * 
     * @param agent Agent对象
     * @param userId 用户ID
     * @return 会话ID
     */
    protected String generateSessionId(Agent agent, String userId) {
        return memoryService.generateSessionId(agent, userId);
    }

    /**
     * 添加用户消息到内存
     * 
     * @param sessionId 会话ID
     * @param userMessage 用户消息
     */
    protected void addUserMessageToMemory(String sessionId, String userMessage) {
        memoryService.addUserMessageToMemory(sessionId, userMessage);
    }

    /**
     * 添加助理消息到内存
     * 
     * @param sessionId 会话ID
     * @param assistantMessage 助理消息
     */
    protected void addAssistantMessageToMemory(String sessionId, String assistantMessage) {
        memoryService.addAssistantMessageToMemory(sessionId, assistantMessage);
    }

    /**
     * 构建Prompt
     * 
     * @param systemPrompt 系统提示词
     * @param historyLength 历史记录长度
     * @param sessionId 会话ID
     * @return Prompt对象
     */
    protected Prompt buildPrompt(String systemPrompt, int historyLength, String sessionId) {
        List<org.springframework.ai.chat.messages.Message> historyMessages = 
            memoryService.getHistoryMessages(sessionId, historyLength);
        
        // 使用智能历史摘要器优化历史消息
        List<org.springframework.ai.chat.messages.Message> summarizedMessages = 
            smartHistorySummarizer.summarize(historyMessages, historyLength);
        
        List<org.springframework.ai.chat.messages.Message> messages = new ArrayList<>();
        messages.add(new org.springframework.ai.chat.messages.SystemMessage(systemPrompt));
        messages.addAll(summarizedMessages);
        return new Prompt(messages);
    }
    
    /**
     * 获取流式模型
     *
     * @param agent Agent对象
     * @return 流式聊天模型
     */
    protected StreamingChatModel getStreamingChatModel(Agent agent) {
        // 参数验证
        if (agent == null) {
            log.warn("getStreamingChatModel参数为空，返回null");
            return null;
        }

        try {
            ChatModel chatModel = agentService.getChatModelForAgent(agent);
            if (chatModel == null) {
                log.warn("无法获取Agent的聊天模型");
                return null;
            }
            
            if (!(chatModel instanceof StreamingChatModel)) {
                log.warn("模型不支持流式输出: {}", chatModel.getClass().getName());
                return null;
            }
            return (StreamingChatModel) chatModel;
        } catch (Exception e) {
            log.error("获取流式模型失败", e);
            return null;
        }
    }
    
    /**
     * 判断异常是否为401未授权错误
     *
     * @param e 异常对象
     * @return 是否为401错误
     */
    protected boolean isUnauthorizedError(Throwable e) {
        return agentErrorHandler.isUnauthorizedError(e);
    }
    
    /**
     * 处理同步处理中的错误
     *
     * @param e 异常对象
     * @param errorMessagePrefix 错误消息前缀
     * @return 错误消息
     */
    protected String handleSyncError(Throwable e, String errorMessagePrefix) {
        return agentErrorHandler.handleSyncError(e, errorMessagePrefix);
    }
    
    /**
     * 获取系统提示词
     * 
     * @param agent Agent对象
     * @return 系统提示词
     */
    protected String getSystemPrompt(Agent agent) {
        return agent.getPromptTemplate() != null ? agent.getPromptTemplate() : DEFAULT_SYSTEM_PROMPT;
    }

    /**
     * 获取历史记录长度
     * 
     * @param agent Agent对象
     * @return 历史记录长度
     */
    protected int getHistoryLength(Agent agent) {
        return agent.getHistoryLength() != null ? agent.getHistoryLength() : 10;
    }
    
    /**
     * 尝试RAG增强
     * 
     * @param agent Agent对象
     * @param userMessage 用户消息
     * @param ragService RAG服务
     * @return RAG响应，如果没有启用RAG或没有结果则返回null
     */
    protected String tryRagEnhancement(Agent agent, String userMessage, RagService ragService) {
        // 参数验证
        if (agent == null) {
            log.warn("Agent对象为空，跳过RAG增强");
            return null;
        }
        
        if (userMessage == null || userMessage.isEmpty()) {
            log.warn("用户消息为空，跳过RAG增强");
            return null;
        }
        
        if (Boolean.TRUE.equals(agent.getEnableRag()) && ragService != null) {
            log.info("Agent启用RAG功能，尝试RAG增强");
            try {
                String ragResponse = ragService.ragQa(agent, userMessage);
                
                // 如果RAG有结果，直接返回
                if (ragResponse != null && !ragResponse.isEmpty()) {
                    log.info("RAG增强返回结果");
                    return saveRagResponseToMemory(agent, ragResponse);
                }
            } catch (Exception e) {
                log.error("RAG增强过程中发生错误", e);
                // RAG失败不应该影响正常的处理流程，继续执行
            }
        }
        return null;
    }
    
    /**
     * 保存RAG响应到内存
     * 
     * @param agent Agent对象
     * @param ragResponse RAG响应
     * @return RAG响应
     */
    protected String saveRagResponseToMemory(Agent agent, String ragResponse) {
        // 参数验证
        if (agent == null) {
            log.warn("Agent对象为空，无法保存RAG响应到内存");
            return ragResponse;
        }
        
        if (ragResponse == null || ragResponse.isEmpty()) {
            log.warn("RAG响应为空，无需保存到内存");
            return ragResponse;
        }
        
        try {
            String sessionId = memoryService.generateSessionId(agent);
            // 将RAG响应添加到ChatMemory
            memoryService.addAssistantMessageToMemory(sessionId, "[RAG增强] " + ragResponse);
            return ragResponse;
        } catch (Exception e) {
            log.error("保存RAG响应到内存时发生错误", e);
            return ragResponse; // 即使保存失败，也要返回原始响应
        }
    }
    
    /**
     * 处理流式响应
     * 职责：统一处理流式响应，包括订阅、错误处理和完成回调
     * 
     * @param tokenConsumer token消费者
     * @param prompt 提示词
     * @param streamingChatModel 流式聊天模型
     * @param sessionId 会话ID
     */
    protected void handleStreamingResponse(Consumer<String> tokenConsumer, Prompt prompt,
                                          StreamingChatModel streamingChatModel, String sessionId) {
        // 参数验证
        if (tokenConsumer == null) {
            log.warn("tokenConsumer不能为空");
            return;
        }
        
        StringBuilder fullText = new StringBuilder();
        AtomicBoolean hasError = new AtomicBoolean(false);
        AtomicBoolean isCompleted = new AtomicBoolean(false);

        try {
            log.debug("开始处理流式响应，Session ID: {}", sessionId);

            if (streamingChatModel != null && prompt != null) {
                log.info("启动流式处理，模型类型: {}, 提示词长度: {}", 
                        streamingChatModel.getClass().getSimpleName(), 
                        prompt.getInstructions().size());
                
                streamingChatModel.stream(prompt)
                    .subscribe(
                        // onNext: 处理每个token
                        chatResponse -> handleToken(chatResponse, tokenConsumer, fullText, isCompleted, hasError),
                        // onError: 处理错误
                        throwable -> handleStreamError(throwable, tokenConsumer, hasError),
                        // onComplete: 处理完成
                        () -> handleStreamComplete(fullText, tokenConsumer, sessionId, isCompleted)
                    );
            } else {
                log.warn("流式模型或提示词为空，无法启动流式处理。模型: {}, 提示词: {}", 
                        streamingChatModel != null, prompt != null);
                handleStreamModelError(tokenConsumer, isCompleted);
            }
        } catch (Exception e) {
            log.error("处理流式响应时发生未预期的异常: {}", e.getMessage(), e);
            handleUnexpectedError(e, tokenConsumer, isCompleted);
        }
    }
    
    /**
     * 处理单个token
     * 职责：处理流式响应中的单个token
     * 
     * @param chatResponse 聊天响应
     * @param tokenConsumer token消费者
     * @param fullText 完整文本构建器
     * @param isCompleted 是否已完成
     * @param hasError 是否有错误
     */
    private void handleToken(org.springframework.ai.chat.model.ChatResponse chatResponse, 
                           Consumer<String> tokenConsumer, StringBuilder fullText, 
                           AtomicBoolean isCompleted, AtomicBoolean hasError) {
        try {
            if (isCompleted.get()) {
                log.trace("流式处理已完成，忽略新的token");
                return;
            }

            // 检查响应是否有效
            if (chatResponse == null || chatResponse.getResult() == null) {
                log.warn("接收到无效的聊天响应");
                return;
            }

            String token = chatResponse.getResult().getOutput().getText();
            if (token != null && !token.isEmpty()) {
                fullText.append(token);
                log.trace("接收到token: length={}, 累计长度: {}", token.length(), fullText.length());

                try {
                    if (tokenConsumer != null) {
                        tokenConsumer.accept(token);
                    }
                } catch (Exception e) {
                    log.error("token消费者处理失败: {}", e.getMessage());
                    hasError.set(true);
                    // 不再重新抛出异常，避免中断流式处理
                    // 但我们应该记录这个错误并向客户端发送错误信息
                    agentErrorHandler.sendErrorMessage(tokenConsumer, "[错误] 处理token时发生错误: " + e.getMessage());
                }
            }
        } catch (Exception e) {
            log.error("处理token时发生错误: {}", e.getMessage(), e);
            hasError.set(true);
        }
    }

    /**
     * 处理流式错误
     * 职责：统一处理流式处理中的错误
     * 
     * @param throwable 异常对象
     * @param tokenConsumer token消费者
     * @param hasError 是否有错误标记
     */
    private void handleStreamError(Throwable throwable, Consumer<String> tokenConsumer, AtomicBoolean hasError) {
        hasError.set(true);
        agentErrorHandler.handleStreamError(throwable, tokenConsumer, "流式调用出错");
    }

    /**
     * 处理流式完成
     * 职责：统一处理流式处理完成后的操作
     * 
     * @param fullText 完整文本
     * @param tokenConsumer token消费者
     * @param sessionId 会话ID
     * @param isCompleted 是否已完成
     */
    private void handleStreamComplete(StringBuilder fullText, Consumer<String> tokenConsumer, 
                                    String sessionId, AtomicBoolean isCompleted) {
        if (isCompleted.getAndSet(true)) {
            log.trace("流式处理已标记为完成，忽略重复的完成回调");
            return;
        }

        // 确保fullText不为null
        if (fullText == null) {
            fullText = new StringBuilder();
        }

        log.info("流式处理完成，总字符数: {}", fullText.length());

        // 添加助理回复到ChatMemory
        if (fullText.length() > 0 && sessionId != null) {
            try {
                addAssistantMessageToMemory(sessionId, fullText.toString());
                if (log.isTraceEnabled()) {
                    log.trace("助理回复已保存到ChatMemory");
                }
            } catch (Exception e) {
                log.error("保存到ChatMemory失败: {}", e.getMessage(), e);
            }
        }

        // 发送完成事件，包含完整内容
        try {
            if (tokenConsumer instanceof TokenConsumerWithCompletion) {
                try {
                    ((TokenConsumerWithCompletion) tokenConsumer).onComplete(fullText.toString());
                } catch (NoClassDefFoundError e) {
                    log.error("TokenConsumerWithCompletion依赖类未找到，跳过完成回调: {}", e.getMessage());
                }
                if (log.isTraceEnabled()) {
                    log.trace("完成事件已发送");
                }
            } else if (tokenConsumer != null) {
                log.warn("tokenConsumer不是TokenConsumerWithCompletion实例，无法发送完成事件");
                // 尽管不是TokenConsumerWithCompletion实例，我们仍然尝试发送一个空的完成信号
                try {
                    tokenConsumer.accept("");
                } catch (Exception e) {
                    log.error("发送完成信号失败: {}", e.getMessage());
                }
            }
        } catch (Exception e) {
            log.error("发送完成事件失败: {}", e.getMessage(), e);
        }
    }

    /**
     * 处理流式模型错误
     * 职责：处理流式模型为空的情况
     * 
     * @param tokenConsumer token消费者
     * @param isCompleted 是否已完成
     */
    private void handleStreamModelError(Consumer<String> tokenConsumer, AtomicBoolean isCompleted) {
        agentErrorHandler.sendErrorMessage(tokenConsumer, "[错误] 流式模型或提示词为空，无法启动流式处理");
        // 标记完成
        isCompleted.set(true);
    }

    /**
     * 处理未预期的错误
     * 职责：处理流式处理中的未预期异常
     * 
     * @param e 异常对象
     * @param tokenConsumer token消费者
     * @param isCompleted 是否已完成
     */
    private void handleUnexpectedError(Exception e, Consumer<String> tokenConsumer, AtomicBoolean isCompleted) {
        String errorMessage = handleSyncError(e, "处理流式响应时发生错误");
        agentErrorHandler.sendErrorMessage(tokenConsumer, "[错误] " + errorMessage);
        // 确保标记为已完成
        isCompleted.set(true);
    }
    
    /**
     * 处理RAG响应的通用逻辑
     * 
     * @param ragResponse RAG响应
     * @param tokenConsumer token消费者（流式处理时使用）
     * @return RAG响应
     */
    protected String handleRagResponse(String ragResponse, Consumer<String> tokenConsumer) {
        if (tokenConsumer != null) {
            // 对于流式处理，我们需要将RAG响应作为token发送
            tokenConsumer.accept(ragResponse);
            // 发送完成信号
            if (tokenConsumer instanceof TokenConsumerWithCompletion) {
                try {
                    ((TokenConsumerWithCompletion) tokenConsumer).onComplete(ragResponse);
                } catch (NoClassDefFoundError e) {
                    log.error("TokenConsumerWithCompletion依赖类未找到，跳过完成回调: {}", e.getMessage());
                }
            }
        }
        return ragResponse;
    }

    /**
     * 处理请求的通用前置逻辑
     * 
     * @param agent Agent对象
     * @param userMessage 用户消息
     * @param userId 用户ID
     * @param ragService RAG服务
     * @param tokenConsumer token消费者（流式处理时使用）
     * @return RAG响应，如果有的话；否则返回null继续正常处理流程
     */
    protected String handlePreProcessing(Agent agent, String userMessage, String userId, RagService ragService, Consumer<String> tokenConsumer) {
        // 为每个用户-Agent组合创建唯一的会话ID
        String sessionId = generateSessionId(agent, userId);

        // 添加用户消息到ChatMemory
        addUserMessageToMemory(sessionId, userMessage);

        // 检查是否启用RAG并尝试RAG增强
        String ragResponse = tryRagEnhancement(agent, userMessage, ragService);
        if (ragResponse != null) {
            log.info("RAG增强返回结果，直接返回");
            return handleRagResponse(ragResponse, tokenConsumer);
        }

        return null;
    }
}