package pangea.hiagent.workpanel;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import pangea.hiagent.web.dto.*;
import pangea.hiagent.web.service.AgentService;
import pangea.hiagent.model.Agent;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.HashSet;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 工作面板服务
 * 负责处理工作面板相关的状态和事件
 */
@Slf4j
@Service
public class WorkPanelService {

    @Autowired
    private AgentService agentService;
    
    // 用于跟踪已发送的事件ID，防止重复发送
    private final Map<String, Set<String>> sentEventIds = new ConcurrentHashMap<>();

    /**
     * 获取工作面板当前状态
     */
    public WorkPanelStatusDto getWorkPanelStatus(String agentId, String userId) {
        try {
            Agent agent = agentService.getAgent(agentId);
            if (agent == null) {
                throw new RuntimeException("Agent不存在");
            }
            
            if (!agent.getOwner().equals(userId)) {
                throw new RuntimeException("无权访问该Agent");
            }
            
            log.info("获取Agent {} 的工作面板状态", agentId);
            // 从工作面板收集器中读取数据
            IWorkPanelDataCollector collector = null; // 移除了对ReActService的依赖
                        
            List<WorkPanelEvent> allEvents = collector != null ? collector.getEvents() : new ArrayList<>();
                        
            // 统计不同类型的事件
            int totalEvents = allEvents.size();
            int successfulCalls = (int) allEvents.stream()
                    .filter(e -> "tool_result".equals(e.getType()) && 
                            (e instanceof ToolEvent && "success".equals(((ToolEvent) e).getToolStatus())))
                    .count();
            int failedCalls = (int) allEvents.stream()
                    .filter(e -> "tool_error".equals(e.getType()) || 
                            ("tool_result".equals(e.getType()) && 
                             (e instanceof ToolEvent && "failure".equals(((ToolEvent) e).getToolStatus()))))
                    .count();
                        
            List<WorkPanelEvent> thinkingEvents = new ArrayList<>();
            List<WorkPanelEvent> toolCallEvents = new ArrayList<>();
            List<WorkPanelEvent> logEvents = new ArrayList<>();
                        
            for (WorkPanelEvent event : allEvents) {
                switch (event.getType()) {
                    case "thought" -> thinkingEvents.add(event);
                    case "tool_call", "tool_result", "tool_error" -> toolCallEvents.add(event);
                    case "log" -> logEvents.add(event);
                    default -> {
                    }
                }
            }
                        
            WorkPanelStatusDto status = WorkPanelStatusDto.builder()
                    .id(agentId + "_workpanel")
                    .agentId(agentId)
                    .agentName(agent.getName())
                    .events(allEvents)
                    .thinkingEvents(thinkingEvents)
                    .toolCallEvents(toolCallEvents)
                    .logEvents(logEvents)
                    .totalEvents(totalEvents)
                    .successfulToolCalls(successfulCalls)
                    .failedToolCalls(failedCalls)
                    .updateTimestamp(System.currentTimeMillis())
                    .isProcessing(false)
                    .build();
            
            return status;
        } catch (Exception e) {
            log.error("获取工作面板状态失败", e);
            throw new RuntimeException("获取工作面板状态失败: " + e.getMessage(), e);
        }
    }

    /**
     * 清空工作面板数据
     */
    public void clearWorkPanel(String agentId, String userId) {
        try {
            Agent agent = agentService.getAgent(agentId);
            if (agent == null) {
                throw new RuntimeException("Agent不存在");
            }
            
            if (!agent.getOwner().equals(userId)) {
                throw new RuntimeException("无权访问该Agent");
            }
            
            log.info("清空Agent {} 的工作面板", agentId);
            // 在实际应用中，这里应该从缓存中清除工作面板数据
            // 清空已发送事件ID跟踪
            sentEventIds.remove(agentId);
        } catch (Exception e) {
            log.error("清空工作面板失败", e);
            throw new RuntimeException("清空工作面板失败: " + e.getMessage(), e);
        }
    }

    /**
     * 生成事件唯一标识
     */
    public String generateEventId(WorkPanelEvent event) {
        if (event == null) {
            return "null_event_" + System.currentTimeMillis();
        }
        
        StringBuilder sb = new StringBuilder();
        sb.append(event.getType()).append("_");
        
        switch (event.getType()) {
            case "thought":
                if (event instanceof ThoughtEvent) {
                    ThoughtEvent thoughtEvent = (ThoughtEvent) event;
                    sb.append(thoughtEvent.getThinkingType() != null ? thoughtEvent.getThinkingType() : "default")
                      .append("_")
                      .append(thoughtEvent.getContent() != null ? thoughtEvent.getContent().hashCode() : 0);
                } else {
                    sb.append("default").append("_").append(0);
                }
                break;
            case "tool_call":
            case "tool_result":
            case "tool_error":
                if (event instanceof ToolEvent) {
                    ToolEvent toolEvent = (ToolEvent) event;
                    sb.append(toolEvent.getToolName() != null ? toolEvent.getToolName() : "unknown")
                      .append("_")
                      .append(toolEvent.getToolAction() != null ? toolEvent.getToolAction() : "unknown")
                      .append("_")
                      .append(event.getTimestamp() != null ? event.getTimestamp() : System.currentTimeMillis());
                } else {
                    sb.append("unknown").append("_").append("unknown").append("_").append(System.currentTimeMillis());
                }
                break;
            case "log":
                if (event instanceof LogEvent) {
                    LogEvent logEvent = (LogEvent) event;
                    sb.append(logEvent.getLogLevel() != null ? logEvent.getLogLevel() : "info")
                      .append("_")
                      .append(logEvent.getContent() != null ? logEvent.getContent().hashCode() : 0);
                } else {
                    sb.append("info").append("_").append(0);
                }
                break;
            case "embed":
                if (event instanceof EmbedEvent) {
                    EmbedEvent embedEvent = (EmbedEvent) event;
                    sb.append(embedEvent.getEmbedTitle() != null ? embedEvent.getEmbedTitle() : "untitled")
                      .append("_")
                      .append(embedEvent.getEmbedUrl() != null ? embedEvent.getEmbedUrl().hashCode() : 0);
                } else {
                    sb.append("untitled").append("_").append(0);
                }
                break;
            default:
                sb.append(event.getTimestamp() != null ? event.getTimestamp() : System.currentTimeMillis());
                break;
        }
        
        return sb.toString();
    }

    /**
     * 检查事件是否已发送
     */
    public boolean isEventAlreadySent(String agentId, WorkPanelEvent event) {
        String eventId = generateEventId(event);
        Set<String> agentEventIds = sentEventIds.computeIfAbsent(agentId, k -> new HashSet<>());
        return !agentEventIds.add(eventId); // 如果已存在，add返回false，表示已发送
    }
}