package pangea.hiagent.web.service;

import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;
import pangea.hiagent.model.Document;
import pangea.hiagent.web.repository.DocumentRepository;

import java.util.List;

/**
 * 文档服务类
 * 提供文档的增删改查功能
 */
@Slf4j
@Service
public class DocumentService extends ServiceImpl<DocumentRepository, Document> {
    
    private final DocumentRepository documentRepository;
    
    public DocumentService(DocumentRepository documentRepository) {
        this.documentRepository = documentRepository;
    }
    
    /**
     * 获取所有文档列表
     * @return 文档列表
     */
    public List<Document> getAllDocuments() {
        try {
            log.info("获取所有文档列表");
            return list();
        } catch (Exception e) {
            log.error("获取文档列表失败", e);
            throw new RuntimeException("获取文档列表失败: " + e.getMessage());
        }
    }
    
    /**
     * 根据ID获取文档
     * @param id 文档ID
     * @return 文档对象
     */
    public Document getDocumentById(String id) {
        try {
            log.info("根据ID获取文档: {}", id);
            return getById(id);
        } catch (Exception e) {
            log.error("获取文档失败，ID: {}", id, e);
            throw new RuntimeException("获取文档失败: " + e.getMessage());
        }
    }
    
    /**
     * 创建文档
     * @param document 文档对象
     * @return 创建后的文档对象
     */
    public Document createDocument(Document document) {
        try {
            log.info("创建文档: {}", document.getName());
            save(document);
            return document;
        } catch (Exception e) {
            log.error("创建文档失败", e);
            throw new RuntimeException("创建文档失败: " + e.getMessage());
        }
    }
    
    /**
     * 更新文档
     * @param document 文档对象
     * @return 更新后的文档对象
     */
    public Document updateDocument(Document document) {
        try {
            log.info("更新文档: {}", document.getId());
            updateById(document);
            return document;
        } catch (Exception e) {
            log.error("更新文档失败，ID: {}", document.getId(), e);
            throw new RuntimeException("更新文档失败: " + e.getMessage());
        }
    }
    
    /**
     * 删除文档
     * @param id 文档ID
     */
    public void deleteDocument(String id) {
        try {
            log.info("删除文档: {}", id);
            removeById(id);
        } catch (Exception e) {
            log.error("删除文档失败，ID: {}", id, e);
            throw new RuntimeException("删除文档失败: " + e.getMessage());
        }
    }
    
    /**
     * 上传文档文件
     * @param file 上传的文件
     * @return 保存后的文档对象
     */
    public Document uploadDocument(MultipartFile file) {
        try {
            log.info("上传文档文件: {}", file.getOriginalFilename());
            
            // 创建文档对象
            Document document = Document.builder()
                .name(file.getOriginalFilename())
                .type(file.getContentType())
                .size(file.getSize())
                .status("uploaded")
                .build();
            
            // 保存文档信息到数据库
            save(document);
            
            return document;
        } catch (Exception e) {
            log.error("上传文档文件失败", e);
            throw new RuntimeException("上传文档文件失败: " + e.getMessage());
        }
    }
}