package pangea.hiagent.agent.react;

import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.chat.client.ChatClient;
import org.springframework.ai.chat.messages.*;
import org.springframework.ai.chat.model.ChatResponse;
import org.springframework.ai.chat.prompt.Prompt;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.context.annotation.Lazy;
import pangea.hiagent.agent.sse.UserSseService;
import pangea.hiagent.tool.impl.HisenseTripTool;
import pangea.hiagent.tool.impl.VisitorAppointmentTool;
import pangea.hiagent.web.service.AgentService;
import pangea.hiagent.web.service.InfoCollectorService;
import pangea.hiagent.workpanel.IWorkPanelDataCollector;
import pangea.hiagent.agent.service.ErrorHandlerService;
import pangea.hiagent.agent.service.TokenConsumerWithCompletion;
import pangea.hiagent.memory.MemoryService;
import pangea.hiagent.model.Agent;
import pangea.hiagent.tool.AgentToolManager;
import pangea.hiagent.tool.impl.DateTimeTools;
import java.util.List;
import java.util.ArrayList;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;

/**
 * 重构后的默认ReAct执行器实现 - 支持真正的流式输出和完整的ReAct流程可观测性
 */
@Slf4j
@Service
public class DefaultReactExecutor implements ReactExecutor {
    
    // 默认系统提示词
    private static final String DEFAULT_SYSTEM_PROMPT = 
        "You are a helpful AI assistant that can use tools to answer questions. " +
        "Use the available tools when needed to gather information. " +
        "Think step by step and show your reasoning process. " +
        "When using tools, clearly indicate your thoughts, actions, and observations.";
    
    private final List<ReactCallback> reactCallbacks = new ArrayList<>();
    private final AtomicInteger stepCounter = new AtomicInteger(0);
    
    @Autowired
    @Lazy
    private IWorkPanelDataCollector workPanelCollector;


    @Autowired
    private DateTimeTools dateTimeTools;
    
    @Autowired
    private MemoryService memoryService;
    
    @Autowired
    private ErrorHandlerService errorHandlerService;

    @Autowired
    private InfoCollectorService infoCollectorService;
    @Autowired
    private AgentService agentService;
    
    private final AgentToolManager agentToolManager;
    @Autowired
    private UserSseService userSseService;

    public DefaultReactExecutor(AgentToolManager agentToolManager) {
        this.agentToolManager = agentToolManager;
    }
    
    /**
     * 添加ReAct回调
     * @param callback ReAct回调
     */
    @Override
    public void addReactCallback(ReactCallback callback) {
        if (callback != null) {
            reactCallbacks.add(callback);
        }
    }
    
    /**
     * 执行ReAct流程（同步方式）
     * @param chatClient ChatClient实例
     * @param userInput 用户输入
     * @param tools 工具列表（已弃用，现在通过Agent获取工具）
     * @return 最终答案
     */
    @Override
    public String execute(ChatClient chatClient, String userInput, List<Object> tools) {
        return executeWithAgent(chatClient, userInput, tools, null);
    }
    
    /**
     * 执行ReAct流程（同步方式）- 支持Agent配置
     * @param chatClient ChatClient实例
     * @param userInput 用户输入
     * @param tools 工具列表（已弃用，现在通过Agent获取工具）
     * @param agent Agent对象（可选）
     * @return 最终答案
     */
    public String executeWithAgent(ChatClient chatClient, String userInput, List<Object> tools, Agent agent) {
        log.info("开始执行ReAct流程，用户输入: {}", userInput);
        
        // 重置步骤计数器
        stepCounter.set(0);
        
        // 获取Agent关联的工具实例
        List<Object> agentTools = getAgentTools(agent);
        
        try {
            // 触发思考步骤
            triggerThinkStep("开始处理用户请求: " + userInput);
            
            // 构建Prompt，包含历史对话记录
            Prompt prompt = buildPromptWithHistory(DEFAULT_SYSTEM_PROMPT, userInput, agent);
            
            // 使用call()获取完整的LLM响应
            // 这会阻塞直到完成整个ReAct循环（思考→行動→观察→...→最终答案）
            log.info("使用call()方法处理ReAct流程，确保完整的工具调用循环");
            
            ChatResponse response = chatClient.prompt(prompt)
                    .tools(agentTools.toArray())
                    .call()
                    .chatResponse();
            
            // 获取响应文本
            String responseText = response.getResult().getOutput().getText();
            
            // 触发观察步骤
            triggerObservationStep(responseText);
            
            // 返回最终结果
            log.info("最终答案: {}", responseText);
            
            // 触发最终答案步骤
            triggerFinalAnswerStep(responseText);
            
            return responseText;
        } catch (Exception e) {
            log.error("执行ReAct流程时发生错误", e);
            return handleReActError(e);
        }
    }
    
    /**
     * 处理ReAct执行错误
     * @param e 异常
     * @return 错误信息
     */
    private String handleReActError(Exception e) {
        return errorHandlerService.handleSyncError(e, "处理ReAct请求时发生错误");
    }
    
    /**
     * 构建包含历史对话记录的Prompt
     * @param systemPrompt 系统提示词
     * @param userInput 用户输入
     * @param agent Agent对象（可选）
     * @return 构建的Prompt
     */
    private Prompt buildPromptWithHistory(String systemPrompt, String userInput, Agent agent) {
        List<org.springframework.ai.chat.messages.Message> messages = new ArrayList<>();
        
        // 添加系统消息
        messages.add(new SystemMessage(systemPrompt));
        
        // 如果提供了Agent，添加历史对话记录
        if (agent != null) {
            try {
                // 生成会话ID
                String sessionId = memoryService.generateSessionId(agent);
                
                // 获取历史记录长度配置，默认为10
                int historyLength = agent.getHistoryLength() != null ? agent.getHistoryLength() : 10;
                
                // 获取历史消息
                List<org.springframework.ai.chat.messages.Message> historyMessages = 
                    memoryService.getHistoryMessages(sessionId, historyLength);
                
                // 添加历史消息到Prompt
                // messages.addAll(historyMessages);
                
                // 将当前用户消息添加到内存中，以便下次对话使用
                memoryService.addUserMessageToMemory(sessionId, userInput);
            } catch (Exception e) {
                log.warn("获取历史对话记录时发生错误: {}", e.getMessage());
            }
        }
        
        // 添加当前用户消息到Prompt
        messages.add(new UserMessage(userInput));
        
        return new Prompt(messages);
    }
    
    /**
     * 流式执行ReAct流程 - 使用真正的流式处理机制
     * 
     * @param chatClient ChatClient实例
     * @param userInput 用户输入
     * @param tools 工具列表（已弃用，现在通过Agent获取工具）
     * @param tokenConsumer token处理回调函数
     */
    @Override
    public void executeStream(ChatClient chatClient, String userInput, List<Object> tools, Consumer<String> tokenConsumer) {
        executeStreamWithAgent(chatClient, userInput, tools, tokenConsumer, null);
    }
    
    /**
     * 流式执行ReAct流程 - 使用真正的流式处理机制（支持Agent配置）
     * 
     * @param chatClient ChatClient实例
     * @param userInput 用户输入
     * @param tools 工具列表（已弃用，现在通过Agent获取工具）
     * @param tokenConsumer token处理回调函数
     * @param agent Agent对象（可选）
     */
    public void executeStreamWithAgent(ChatClient chatClient, String userInput, List<Object> tools, Consumer<String> tokenConsumer, Agent agent) {
        log.info("使用stream()方法处理ReAct流程，支持真正的流式输出");
        
        // 重置步骤计数器
        stepCounter.set(0);
        
        // 获取Agent关联的工具实例
        List<Object> agentTools = getAgentTools(agent);
        
        // 使用StringBuilder累积完整响应
        StringBuilder fullResponse = new StringBuilder();
        
        try {
            // 触发思考步骤
            triggerThinkStep("开始处理用户请求: " + userInput);
            
            // 构建Prompt，包含历史对话记录
            Prompt prompt = buildPromptWithHistory(DEFAULT_SYSTEM_PROMPT, userInput, agent);
            VisitorAppointmentTool hisenseTripTool = new VisitorAppointmentTool(agentService,infoCollectorService,userSseService);
            hisenseTripTool.initialize();
            // 订阅流式响应
            chatClient.prompt(prompt)
                    .tools(hisenseTripTool)
                    .stream()
                    .chatResponse()
                    .subscribe(
                        chatResponse -> handleTokenResponse(chatResponse, tokenConsumer, fullResponse),
                        throwable -> handleStreamError(throwable, tokenConsumer),
                        () -> handleStreamCompletion(tokenConsumer, fullResponse, agent)
                    );
            
        } catch (Exception e) { 
            log.error("流式执行ReAct流程时发生错误", e);
            errorHandlerService.handleReactFlowError(e, tokenConsumer);
        }
    }
    
    /**
     * 处理token响应
     * 
     * @param chatResponse 聊天响应
     * @param tokenConsumer token处理回调函数
     * @param fullResponse 完整响应构建器
     */
    private void handleTokenResponse(org.springframework.ai.chat.model.ChatResponse chatResponse, Consumer<String> tokenConsumer, StringBuilder fullResponse) {
        try {
            // 获取token
            String token = chatResponse.getResult().getOutput().getText();
            
            // 验证token是否有效
            if (isValidToken(token)) {
                // 累积完整响应
                fullResponse.append(token);
                
                // 分析token内容，识别工具调用和结果
                analyzeAndRecordToolEvents(token, fullResponse.toString());
                
                // 实时发送token给客户端
                if (tokenConsumer != null) {
                    tokenConsumer.accept(token);
                }
                
                // 记录思考过程
                processTokenForSteps(token);
            }
        } catch (Exception e) {
            log.error("处理token时发生错误", e);
        }
    }
    
    /**
     * 处理流式完成
     * 
     * @param tokenConsumer token处理回调函数
     * @param fullResponse 完整响应构建器
     * @param agent Agent对象
     */
    private void handleStreamCompletion(Consumer<String> tokenConsumer, StringBuilder fullResponse, Agent agent) {
        try {
            log.info("流式处理完成");
            // 触发最终答案步骤
            triggerFinalAnswerStep(fullResponse.toString());
            
            // 将助理回复添加到ChatMemory
            saveAssistantResponseToMemory(agent, fullResponse.toString());
            
            // 发送完成事件，包含完整内容
            sendCompletionEvent(tokenConsumer, fullResponse.toString());
        } catch (Exception e) {
            log.error("处理流式完成回调时发生错误", e);
            // 即使在完成回调中出现错误，也要确保标记完成
            handleCompletionError(tokenConsumer, e);
        }
    }
    
    /**
     * 保存助理回复到内存
     * 
     * @param agent Agent对象
     * @param response 助理回复
     */
    private void saveAssistantResponseToMemory(Agent agent, String response) {
        if (agent != null) {
            try {
                String sessionId = memoryService.generateSessionId(agent);
                memoryService.addAssistantMessageToMemory(sessionId, response);
            } catch (Exception e) {
                log.warn("保存助理回复到内存时发生错误: {}", e.getMessage());
            }
        }
    }
    
    /**
     * 处理完成错误
     * 
     * @param tokenConsumer token处理回调函数
     * @param e 异常
     */
    private void handleCompletionError(Consumer<String> tokenConsumer, Exception e) {
        if (tokenConsumer instanceof TokenConsumerWithCompletion) {
            try {
                String errorId = errorHandlerService.generateErrorId();
                String fullErrorMessage = errorHandlerService.buildFullErrorMessage("处理完成时发生错误", e, errorId, "ReAct");
                ((TokenConsumerWithCompletion) tokenConsumer).onComplete("[" + errorId + "] " + fullErrorMessage);
            } catch (Exception ex) {
                log.error("调用onComplete时发生错误", ex);
            }
        }
    }
    
    /**
     * 检查token是否有效
     * @param token token字符串
     * @return 是否有效
     */
    private boolean isValidToken(String token) {
        return token != null && !token.isEmpty();
    }
    
    /**
     * 处理token以识别不同的步骤类型
     * @param token token字符串
     */
    private void processTokenForSteps(String token) {
        // 参数验证
        if (token == null || token.isEmpty()) {
            log.debug("接收到空的token，跳过处理");
            return;
        }
        
        if (token.contains("Thought:")) {
            triggerThinkStep(token);
        } else if (token.contains("Action:")) {
            // 提取工具名称和参数
            String toolName = extractToolName(token);
            Object toolArgs = extractToolArgs(token);
            triggerActionStep(toolName != null ? toolName : "unknown tool", token, toolArgs);
        } else if (token.contains("Observation:")) {
            triggerObservationStep(token);
        }
    }
    
    /**
     * 处理流式处理错误
     * 
     * @param throwable 异常对象
     * @param tokenConsumer token消费者
     */
    private void handleStreamError(Throwable throwable, Consumer<String> tokenConsumer) {
        errorHandlerService.handleStreamError(throwable, tokenConsumer, "ReAct流式处理");
    }
    
    /**
     * 发送完成事件
     * @param tokenConsumer token消费者
     * @param fullResponse 完整响应
     */
    private void sendCompletionEvent(Consumer<String> tokenConsumer, String fullResponse) {
        // 参数验证
        if (fullResponse == null) {
            fullResponse = "";
        }
        
        if (tokenConsumer instanceof TokenConsumerWithCompletion) {
            log.debug("调用onComplete，内容长度: {}", fullResponse.length());
            ((TokenConsumerWithCompletion) tokenConsumer).onComplete(fullResponse);
        } else if (tokenConsumer != null) {
            log.warn("tokenConsumer不是TokenConsumerWithCompletion实例");
            tokenConsumer.accept("");
        }
    }
    
    /**
     * 分析token内容，识别工具调用和结果
     * 这个方法通过分析响应中的特殊标记来识别工具调用
     * 
     * @param token 当前token
     * @param fullResponse 完整响应
     */
    private void analyzeAndRecordToolEvents(String token, String fullResponse) {
        if (!isValidToken(token) || workPanelCollector == null) {
            return;
        }
        
        try {
            // 检查工具调用的标记
            // 通常格式为: "Tool: [工具名称]" 或 "Calling [工具名称]" 或类似的模式
            if (isToolCall(token)) {
                String toolName = extractToolName(token);
                if (isValidToolName(toolName)) {
                    // 记录工具调用开始
                    workPanelCollector.recordToolCallAction(toolName, extractToolArgs(token), null, "pending", null);
                }
            }
            
            // 检查工具结果的标记
            else if (isToolResult(token)) {
                String toolName = extractToolName(fullResponse);
                String result = extractToolResult(token);
                if (isValidToolName(toolName)) {
                    // 记录工具调用完成
                    workPanelCollector.recordToolCallAction(toolName, extractToolArgs(token), result, "success", null);
                }
            }
            
            // 检查错误标记
            else if (isError(token)) {
                String toolName = extractToolName(fullResponse);
                if (isValidToolName(toolName)) {
                    // 记录工具调用错误
                    workPanelCollector.recordToolCallAction(toolName, extractToolArgs(token), null, "error", null);
                }
            }
        } catch (Exception e) {
            log.debug("分析工具调用事件时发生错误: {}", e.getMessage());
        }
    }
    
    /**
     * 检查是否为工具调用
     * @param token token字符串
     * @return 是否为工具调用
     */
    private boolean isToolCall(String token) {
        return token != null && (token.contains("Tool:") || token.contains("Calling") || 
               token.contains("tool:") || token.contains("calling"));
    }
    
    /**
     * 检查是否为工具结果
     * @param token token字符串
     * @return 是否为工具结果
     */
    private boolean isToolResult(String token) {
        return token != null && (token.contains("Result:") || token.contains("result:") || 
               token.contains("Output:") || token.contains("output:"));
    }
    
    /**
     * 检查是否为错误信息
     * @param token token字符串
     * @return 是否为错误信息
     */
    private boolean isError(String token) {
        return token != null && (token.contains("Error:") || token.contains("error:") || 
               token.contains("Exception:") || token.contains("exception:"));
    }
    
    /**
     * 检查工具名称是否有效
     * @param toolName 工具名称
     * @return 是否有效
     */
    private boolean isValidToolName(String toolName) {
        return toolName != null && !toolName.isEmpty();
    }
    
    /**
     * 从文本中提取工具名称
     * 
     * @param text 输入文本
     * @return 工具名称，如果未找到则返回null
     */
    private String extractToolName(String text) {
        if (text == null) return null;
        
        // 尝试从常见的工具调用格式中提取工具名称
        String[] patterns = {
            "Tool: (\\w+)",
            "tool: (\\w+)",
            "Calling (\\w+)",
            "calling (\\w+)",
            "Use (\\w+)",
            "use (\\w+)",
            "Tool\\((\\w+)\\)",
            "tool\\((\\w+)\\)"
        };
        
        for (String pattern : patterns) {
            java.util.regex.Pattern p = java.util.regex.Pattern.compile(pattern);
            java.util.regex.Matcher m = p.matcher(text);
            if (m.find()) {
                return m.group(1);
            }
        }
        
        return null;
    }
    
    /**
     * 从文本中提取工具参数
     * 
     * @param text 输入文本
     * @return 工具参数，如果未找到则返回null
     */
    private Object extractToolArgs(String text) {
        if (text == null) return null;
        
        // 简单的参数提取逻辑
        // 可以根据具体的工具调用格式进行更复杂的解析
        java.util.Map<String, Object> args = new java.util.HashMap<>();
        
        // 尝试提取括号内的内容作为参数
        java.util.regex.Pattern pattern = java.util.regex.Pattern.compile("\\(([^)]*)\\)");
        java.util.regex.Matcher matcher = pattern.matcher(text);
        if (matcher.find()) {
            args.put("params", matcher.group(1));
            return args;
        }
        
        return args.isEmpty() ? null : args;
    }
    
    /**
     * 从token中提取工具结果
     * 
     * @param token 输入token
     * @return 工具结果
     */
    private String extractToolResult(String token) {
        if (token == null) return "";
        
        // 简单的结果提取逻辑
        // 提取冒号后的内容
        int colonIndex = token.lastIndexOf(':');
        if (colonIndex >= 0 && colonIndex < token.length() - 1) {
            return token.substring(colonIndex + 1).trim();
        }
        
        return token.trim();
    }
    
    /**
     * 触发思考步骤
     * @param content 思考内容
     */
    private void triggerThinkStep(String content) {
        int stepNumber = stepCounter.incrementAndGet();
        ReactStep reactStep = new ReactStep(stepNumber, ReactStepType.THOUGHT, content);
        notifyCallbacks(reactStep);
    }
    
    /**
     * 触发行动步骤
     * @param toolName 工具名称
     * @param toolAction 工具行动
     * @param toolArgs 工具参数
     */
    private void triggerActionStep(String toolName, String toolAction, Object toolArgs) {
        int stepNumber = stepCounter.incrementAndGet();
        ReactStep reactStep = new ReactStep(stepNumber, ReactStepType.ACTION, "执行工具: " + toolName);
        
        ReactStep.ToolCallAction toolActionObj = new ReactStep.ToolCallAction(toolName, toolArgs);
        reactStep.setAction(toolActionObj);
        
        notifyCallbacks(reactStep);
    }
    
    /**
     * 触发观察步骤
     * @param observation 观察内容
     */
    private void triggerObservationStep(String observation) {
        int stepNumber = stepCounter.incrementAndGet();
        ReactStep reactStep = new ReactStep(stepNumber, ReactStepType.OBSERVATION, observation);
        
        ReactStep.ToolObservation toolObservation = new ReactStep.ToolObservation(observation);
        reactStep.setObservation(toolObservation);
        
        notifyCallbacks(reactStep);
    }
    
    /**
     * 触发最终答案步骤
     * @param finalAnswer 最终答案
     */
    private void triggerFinalAnswerStep(String finalAnswer) {
        int stepNumber = stepCounter.incrementAndGet();
        ReactStep reactStep = new ReactStep(stepNumber, ReactStepType.FINAL_ANSWER, finalAnswer);
        notifyCallbacks(reactStep);
    }
    
    /**
     * 通知所有回调
     * @param reactStep ReAct步骤
     */
    private void notifyCallbacks(ReactStep reactStep) {
        for (ReactCallback callback : reactCallbacks) {
            try {
                callback.onStep(reactStep);
            } catch (Exception e) {
                log.error("执行ReAct回调时发生错误", e);
            }
        }
    }
    
    /**
     * 获取Agent关联的工具实例
     * @param agent Agent对象
     * @return 工具实例列表
     */
    private List<Object> getAgentTools(Agent agent) {
        if (agent == null) {
            log.debug("Agent为空，返回空工具列表");
            return new ArrayList<>();
        }
        
        try {
            List<Object> tools = agentToolManager.getAvailableToolInstances(agent);
            tools.add(dateTimeTools);
            log.debug("获取到Agent '{}' 的工具实例数量: {}", agent.getName(), tools.size());
            return tools;
        } catch (Exception e) {
            log.error("获取Agent工具实例时发生错误", e);
            return new ArrayList<>();
        }
    }
}