package pangea.hiagent.agent.service;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.web.servlet.mvc.method.annotation.SseEmitter;
import pangea.hiagent.model.Agent;
import pangea.hiagent.model.AgentDialogue;
import pangea.hiagent.common.utils.ValidationUtils;
import pangea.hiagent.agent.processor.AgentProcessor;
import pangea.hiagent.agent.sse.UserSseService;
import pangea.hiagent.common.utils.LogUtils;
import pangea.hiagent.common.utils.UserUtils;
import pangea.hiagent.web.dto.AgentRequest;
import pangea.hiagent.web.service.AgentService;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * 完成回调处理服务
 * 负责处理流式输出完成后的回调操作
 */
@Slf4j
@Service
public class CompletionHandlerService {
    
    @Autowired
    private AgentService agentService;
    
    @Autowired
    private UserSseService unifiedSseService;
    
    @Autowired
    private ErrorHandlerService errorHandlerService;
    
    /**
     * 处理完成回调
     *
     * @param emitter SSE发射器
     * @param processor Agent处理器
     * @param agent Agent对象
     * @param request Agent请求
     * @param userId 用户ID
     * @param fullContent 完整内容
     * @param isCompleted 完成状态标记
     */
    public void handleCompletion(SseEmitter emitter, AgentProcessor processor, Agent agent, 
                               AgentRequest request, String userId, 
                               String fullContent, AtomicBoolean isCompleted) {
        LogUtils.enterMethod("handleCompletion", emitter, processor, agent, request, userId);
        // 参数验证
        if (ValidationUtils.isNull(emitter, "emitter")) {
            log.error("SSE发射器不能为空");
            LogUtils.exitMethod("handleCompletion", "SSE发射器不能为空");
            return;
        }
        
        if (ValidationUtils.isNull(processor, "processor")) {
            log.error("Agent处理器不能为空");
            LogUtils.exitMethod("handleCompletion", "Agent处理器不能为空");
            return;
        }
        
        if (ValidationUtils.isNull(agent, "agent")) {
            log.error("Agent对象不能为空");
            LogUtils.exitMethod("handleCompletion", "Agent对象不能为空");
            return;
        }
        
        if (ValidationUtils.isNull(request, "request")) {
            log.error("Agent请求不能为空");
            LogUtils.exitMethod("handleCompletion", "Agent请求不能为空");
            return;
        }
        
        if (ValidationUtils.isBlank(userId, "userId")) {
            log.error("用户ID不能为空");
            LogUtils.exitMethod("handleCompletion", "用户ID不能为空");
            return;
        }
        
        if (ValidationUtils.isNull(isCompleted, "isCompleted")) {
            log.error("完成状态标记不能为空");
            LogUtils.exitMethod("handleCompletion", "完成状态标记不能为空");
            return;
        }

        log.info("{} Agent处理完成，总字符数: {}", processor.getProcessorType(), fullContent != null ? fullContent.length() : 0);

        // 发送完成事件
        Exception completionException = null;
        try {
            // 发送完整内容作为最后一个token
            // if (fullContent != null && !fullContent.isEmpty()) {
            //     eventService.sendTokenEvent(emitter, fullContent);
            // }
            // 发送完成信号
            emitter.send("[DONE]");
        } catch (Exception e) {
            log.error("发送完成信号失败", e);
            completionException = e;
        }
        
        // 保存对话记录
        try {
            saveDialogue(agent, request, userId, fullContent);
            log.info("对话记录保存成功");
        } catch (Exception e) {
            log.error("保存对话记录失败", e);
            // 记录异常但不中断流程，继续关闭emitter
            completionException = e;
        }
        
        // 最后才关闭emitter，确保所有操作都完成后再提交响应
        try {
            // 检查emitter是否已经完成，避免重复关闭
            if (!unifiedSseService.isEmitterCompleted(emitter)) {
                unifiedSseService.completeEmitter(emitter, isCompleted);
                log.debug("SSE Emitter已关闭");
            } else {
                log.debug("SSE Emitter已完成，跳过关闭操作");
            }
        } catch (Exception e) {
            log.error("关闭Emitter时发生错误", e);
        }
        
        LogUtils.exitMethod("handleCompletion", "处理完成");
    }
    
    /**
     * 保存对话记录
     */
    public void saveDialogue(Agent agent, AgentRequest request, String userId, String responseContent) {
        LogUtils.enterMethod("saveDialogue", agent, request, userId);
        // 参数验证
        if (ValidationUtils.isNull(agent, "agent")) {
            log.error("Agent对象不能为空");
            LogUtils.exitMethod("saveDialogue", "Agent对象不能为空");
            return;
        }
        
        if (ValidationUtils.isNull(request, "request")) {
            log.error("Agent请求不能为空");
            LogUtils.exitMethod("saveDialogue", "Agent请求不能为空");
            return;
        }
        
        if (ValidationUtils.isBlank(userId, "userId")) {
            log.error("用户ID不能为空");
            LogUtils.exitMethod("saveDialogue", "用户ID不能为空");
            return;
        }
        
        try {
            // 创建对话记录
            AgentDialogue dialogue = AgentDialogue.builder()
                .agentId(request.getAgentId())
                .userMessage(request.getUserMessage())
                .agentResponse(responseContent)
                .userId(userId)
                .build();
            
            // 确保ID被设置
            if (dialogue.getId() == null || dialogue.getId().isEmpty()) {
                dialogue.setId(java.util.UUID.randomUUID().toString());
            }
            
            // 设置创建人和更新人信息
            // 在异步线程中获取用户ID
            String currentUserId = UserUtils.getCurrentUserIdInAsync();
            if (currentUserId == null) {
                currentUserId = userId; // 回退到传入的userId
            }
            dialogue.setCreatedBy(currentUserId);
            dialogue.setUpdatedBy(currentUserId);

            // 保存对话记录
            agentService.saveDialogue(dialogue);
            LogUtils.exitMethod("saveDialogue", "保存成功");
        } catch (Exception e) {
            log.error("保存对话记录失败", e);
            LogUtils.exitMethod("saveDialogue", e);
            throw new RuntimeException("保存对话记录失败", e);
        }
    }
}