package pangea.hiagent.tool.impl;

import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;
import org.springframework.ai.tool.annotation.Tool;
import pangea.hiagent.tool.BaseTool;

import java.time.LocalDateTime;
import java.time.LocalDate;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;

/**
 * 日期时间工具类
 * 提供日期和时间相关的功能
 */
@Slf4j
@Component
public class DateTimeTools extends BaseTool {
    
    private String dateTimeFormat = "yyyy-MM-dd HH:mm:ss";
    
    private String dateFormat = "yyyy-MM-dd";
    
    private String timeFormat = "HH:mm:ss";
    
    @Tool(description = "获取当前日期和时间，返回格式为 'yyyy-MM-dd HH:mm:ss'")
    public String getCurrentDateTime() {
        return execute("getCurrentDateTime", () -> {
            try {
                if (dateTimeFormat == null || dateTimeFormat.trim().isEmpty()) {
                    dateTimeFormat = "yyyy-MM-dd HH:mm:ss";
                }
                String dateTime = LocalDateTime.now().format(DateTimeFormatter.ofPattern(dateTimeFormat));
                log.info("【时间工具】获取当前日期时间: {}", dateTime);
                return dateTime;
            } catch (Exception e) {
                log.error("获取当前日期时间时发生错误: {}", e.getMessage(), e);
                // 发生错误时回退到默认格式
                return LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss"));
            }
        });
    }
    
    @Tool(description = "获取当前日期，返回格式为 'yyyy-MM-dd'")
    public String getCurrentDate() {
        return execute("getCurrentDate", () -> {
            try {
                if (dateFormat == null || dateFormat.trim().isEmpty()) {
                    dateFormat = "yyyy-MM-dd";
                }
                String date = LocalDate.now().format(DateTimeFormatter.ofPattern(dateFormat));
                log.info("【时间工具】获取当前日期: {}", date);
                return date;
            } catch (Exception e) {
                log.error("获取当前日期时发生错误: {}", e.getMessage(), e);
                // 发生错误时回退到默认格式
                return LocalDate.now().format(DateTimeFormatter.ofPattern("yyyy-MM-dd"));
            }
        });
    }
    
    @Tool(description = "获取当前时间，返回格式为 'HH:mm:ss'")
    public String getCurrentTime() {
        return execute("getCurrentTime", () -> {
            try {
                if (timeFormat == null || timeFormat.trim().isEmpty()) {
                    timeFormat = "HH:mm:ss";
                }
                String time = LocalTime.now().format(DateTimeFormatter.ofPattern(timeFormat));
                log.info("【时间工具】获取当前时间: {}", time);
                return time;
            } catch (Exception e) {
                log.error("获取当前时间时发生错误: {}", e.getMessage(), e);
                // 发生错误时回退到默认格式
                return LocalTime.now().format(DateTimeFormatter.ofPattern("HH:mm:ss"));
            }
        });
    }
    
    @Tool(description = "获取当前时间戳（毫秒），返回自1970年1月1日00:00:00 UTC以来的毫秒数")
    public String getCurrentTimeMillis() {
        return execute("getCurrentTimeMillis", () -> {
            try {
                long timestamp = System.currentTimeMillis();
                log.info("【时间工具】获取当前时间戳: {}", timestamp);
                return String.valueOf(timestamp);
            } catch (Exception e) {
                log.error("获取当前时间戳时发生错误: {}", e.getMessage(), e);
                return String.valueOf(System.currentTimeMillis());
            }
        });
    }
}
