package pangea.hiagent.tool;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Service;
import org.springframework.aop.support.AopUtils;
import pangea.hiagent.model.Agent;
import pangea.hiagent.model.Tool;
import pangea.hiagent.web.repository.AgentToolRelationRepository;
import pangea.hiagent.web.service.ToolService;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * Agent工具管理服务类
 * 负责管理Agent可用的工具列表
 */
@Slf4j
@Service
public class AgentToolManager {
    
    @Autowired
    private ToolService toolService;
    
    @Autowired
    private ApplicationContext applicationContext;
    
    @Autowired
    private AgentToolRelationRepository agentToolRelationRepository;
    
    /**
     * 获取Agent可用的工具列表
     * @param agent Agent对象
     * @return 工具列表
     */
    public List<Tool> getAvailableTools(Agent agent) {
        try {
            log.info("获取Agent可用工具列表，Agent ID: {}, 名称: {}", agent.getId(), agent.getName());
            
            // 获取与Agent关联的Tool ID列表
            List<String> toolIds = agentToolRelationRepository.getToolIdsByAgentId(agent.getId());
            log.info("Agent关联的工具ID数量: {}", toolIds != null ? toolIds.size() : 0);
            
            if (toolIds == null || toolIds.isEmpty()) {
                // 如果没有关联特定工具，则返回该用户的所有活跃工具
                List<Tool> allTools = toolService.getUserToolsByStatus(agent.getOwner(), "active");
                log.info("返回用户所有活跃工具，数量: {}", allTools != null ? allTools.size() : 0);
                return allTools != null ? allTools : List.of();
            }
            
            // 根据Tool ID获取具体的Tool对象
            List<Tool> tools = new ArrayList<>();
            for (String toolId : toolIds) {
                Tool tool = toolService.getById(toolId);
                if (tool != null) {
                    tools.add(tool);
                }
            }
            
            log.info("获取到的具体工具数量: {}", tools.size());
            tools.forEach(tool -> log.info("工具名称: {}", tool.getName()));

            return tools;
        } catch (Exception e) {
            log.error("获取Agent可用工具时发生错误", e);
            return List.of();
        }
    }
    
    /**
     * 获取Agent关联的工具名称列表
     * @param agent Agent对象
     * @return 工具名称列表
     */
    public List<String> getToolNames(Agent agent) {
        List<Tool> tools = getAvailableTools(agent);
        return tools.stream().map(Tool::getName).collect(Collectors.toList());
    }
    
    /**
     * 根据工具名称筛选工具
     * @param allTools 所有工具
     * @param toolNames 工具名称列表
     * @return 筛选后的工具列表
     */
    public List<Tool> filterToolsByName(List<Tool> allTools, List<String> toolNames) {
        return allTools.stream()
            .filter(tool -> toolNames.contains(tool.getName()))
            .collect(Collectors.toList());
    }
    
    /**
     * 根据工具名称集合筛选工具实例（用于ReActService）
     * @param allTools 所有工具实例
     * @param toolNames 工具名称集合
     * @return 筛选后的工具实例列表
     */
    public List<Object> filterToolsByInstances(List<Object> allTools, Set<String> toolNames) {
        log.debug("开始筛选工具实例，工具名称集合: {}", toolNames);
        
        if (toolNames == null || toolNames.isEmpty()) {
            log.debug("工具名称集合为空，返回所有工具实例");
            return allTools;
        }
        
        List<Object> filteredTools = allTools.stream()
            .filter(tool -> {
                // 获取工具类名（不含包名）
                String className = tool.getClass().getSimpleName();
                log.debug("检查工具类: {}", className);
                
                // 检查类名是否匹配
                boolean isMatch = toolNames.contains(className) ||
                    toolNames.stream().anyMatch(name ->
                        className.toLowerCase().contains(name.toLowerCase()));
                
                if (isMatch) {
                    log.debug("工具 {} 匹配成功", className);
                }
                
                return isMatch;
            })
            .collect(Collectors.toList());
            
        log.debug("筛选完成，返回 {} 个工具实例", filteredTools.size());
        return filteredTools;
    }
    
    /**
     * 构建工具描述文本
     * @param tools 工具列表
     * @return 工具描述文本
     */
    public String buildToolsDescription(List<Tool> tools) {
        if (tools.isEmpty()) {
            return "（暂无可用工具）";
        }
        
        StringBuilder description = new StringBuilder();
        for (int i = 0; i < tools.size(); i++) {
            Tool tool = tools.get(i);
            description.append(i + 1).append(". ");
            description.append(tool.getName());
            if (hasValue(tool.getDisplayName())) {
                description.append(" - ").append(tool.getDisplayName());
            }
            if (hasValue(tool.getDescription())) {
                description.append(" - ").append(tool.getDescription());
            }
            description.append("\n");
        }
        
        return description.toString();
    }
    
    /**
     * 检查字符串是否有值
     * @param value 字符串值
     * @return 是否有值
     */
    private boolean hasValue(String value) {
        return value != null && !value.isEmpty();
    }
    
    /**
     * 获取Bean的原始目标类（穿透Spring AOP代理）
     * 
     * 用于处理以下场景：
     * 1. Bean被Spring AOP代理，需要获取原始类信息
     * 2. 获取原始类的方法和字段信息
     * 3. 进行类型检查和反射操作
     * 
     * @param bean Bean实例（可能是代理对象）
     * @return 原始目标类的Class对象
     */
    private Class<?> getTargetClass(Object bean) {
        if (bean == null) {
            return null;
        }
        return AopUtils.getTargetClass(bean);
    }
    
    /**
     * 根据原始类名获取工具实例（处理AOP代理）
     * 
     * 此方法用于以下场景：
     * 1. 需要根据原始类名精确查找工具实例
     * 2. Bean被Spring AOP代理，但需要基于原始类名进行查找
     * 3. 支持类名模糊匹配和精确匹配
     * 
     * 匹配规则：
     * - 精确匹配：原始类名完全相同
     * - 模糊匹配：原始类名包含指定的部分（不区分大小写）
     * 
     * @param agent Agent对象
     * @param originalClassName 原始类名（可以是完整类名或简单类名）
     * @param isExactMatch 是否精确匹配（true为精确，false为包含匹配）
     * @return 匹配的工具实例列表
     */
    public List<Object> getToolInstancesByOriginalClassName(Agent agent, String originalClassName, boolean isExactMatch) {
        List<Object> result = new ArrayList<>();
        
        if (agent == null || originalClassName == null || originalClassName.trim().isEmpty()) {
            log.warn("参数校验失败: agent={}, originalClassName={}", agent != null ? agent.getId() : null, originalClassName);
            return result;
        }
        
        try {
            log.debug("[{}] 根据原始类名'{}' 查找工具实例，精确匹配: {}", agent.getName(), originalClassName, isExactMatch);
            
            List<Tool> availableTools = getAvailableTools(agent);
            
            for (Tool tool : availableTools) {
                try {
                    if (tool.getBeanName() == null || tool.getBeanName().trim().isEmpty()) {
                        continue;
                    }
                    
                    Object bean = null;
                    try {
                        bean = applicationContext.getBean(tool.getBeanName());
                    } catch (Exception e) {
                        log.debug("[{}] 工具'{}' 的Bean查找失败，跳过", agent.getName(), tool.getName());
                        continue;
                    }
                    
                    if (bean == null) {
                        continue;
                    }
                    
                    // 获取原始目标类
                    Class<?> targetClass = getTargetClass(bean);
                    if (targetClass == null) {
                        targetClass = bean.getClass();
                    }
                    
                    String targetClassName = targetClass.getSimpleName();
                    String targetFullClassName = targetClass.getName();
                    
                    // 根据匹配模式进行判断
                    boolean matches = false;
                    if (isExactMatch) {
                        // 精确匹配：检查简单类名和完整类名
                        matches = originalClassName.equals(targetClassName) || 
                                 originalClassName.equals(targetFullClassName);
                    } else {
                        // 模糊匹配：检查是否包含（不区分大小写）
                        matches = targetClassName.toLowerCase().contains(originalClassName.toLowerCase()) ||
                                 targetFullClassName.toLowerCase().contains(originalClassName.toLowerCase());
                    }
                    
                    if (matches) {
                        result.add(bean);
                        log.debug("[{}] 根据原始类名'{}' 匹配到工具实例: {}", agent.getName(), originalClassName, targetClassName);
                    }
                } catch (Exception e) {
                    log.debug("[{}] 处理工具'{}' 时出错", agent.getName(), tool.getName(), e);
                }
            }
            
            log.debug("[{}] 根据原始类名'{}' 共找到 {} 个工具实例", agent.getName(), originalClassName, result.size());
        } catch (Exception e) {
            log.error("[{}] 根据原始类名查找工具实例时发生错误", agent.getName(), e);
        }
        
        return result;
    }
    
    /**
     * 根据原始类名精确获取单个工具实例（处理AOP代理）
     * 
     * @param agent Agent对象
     * @param originalClassName 原始类名
     * @return 匹配的工具实例，如果没有找到返回null，如果找到多个返回第一个
     */
    public Object getToolInstanceByOriginalClassName(Agent agent, String originalClassName) {
        List<Object> instances = getToolInstancesByOriginalClassName(agent, originalClassName, true);
        return instances.isEmpty() ? null : instances.get(0);
    }
    
    /**
     * 检查工具实例是否属于指定的原始类（处理AOP代理）
     * 
     * @param toolInstance 工具实例
     * @param originalClassName 原始类名
     * @return 如果工具实例属于指定的原始类返回true，否则返回false
     */
    public boolean isToolInstanceOfClass(Object toolInstance, String originalClassName) {
        if (toolInstance == null || originalClassName == null || originalClassName.trim().isEmpty()) {
            return false;
        }
        
        Class<?> targetClass = getTargetClass(toolInstance);
        if (targetClass == null) {
            targetClass = toolInstance.getClass();
        }
        
        return originalClassName.equals(targetClass.getSimpleName()) ||
               originalClassName.equals(targetClass.getName());
    }
    /**
     * 根据Agent获取可用的工具实例
     * 
     * 实现方案: 遍历beanName查找，效率高且可靠
     * 优势: 
     *   1. 逐个beanName查找，效率高且可靠
     *   2. 配置化手段，不依赖于字段匹配
     *   3. 支持Spring AOP代理，通过穿透代理获取原始类信息
     *   4. 预留异常处理方案，处理部分实例无法初始化的情况
     * 
     * @param agent Agent对象
     * @return 工具实例列表（包含AOP代理后的实例）
     */
    public List<Object> getAvailableToolInstances(Agent agent) {
        try {
            log.info("[{}] 开始获取可用的工具实例", agent.getName());
            
            // 获取Agent可用的工具定义
            List<Tool> availableTools = getAvailableTools(agent);
            log.debug("[{}] 获取到了{}个工具定义", agent.getName(), availableTools.size());
            
            List<Object> toolInstances = new ArrayList<>();
            List<String> failedBeans = new ArrayList<>();
            
            // 遍历每个工具定义，根据beanName查找Spring Bean实例
            for (Tool tool : availableTools) {
                try {
                    // 验证beanName是否为空
                    if (tool.getBeanName() == null || tool.getBeanName().trim().isEmpty()) {
                        log.warn("[{}] 工具'{}' 没有配置beanName，跳过此工具", agent.getName(), tool.getName());
                        failedBeans.add(tool.getName() + " (beanName为null)");
                        continue;
                    }
                    
                    // 根据beanName查找Bean实例
                    Object bean = null;
                    try {
                        bean = applicationContext.getBean(tool.getBeanName());
                    } catch (Exception e) {
                        log.warn("[{}] 工具'{}' 查找Bean'{}' 失败，错误消息: {}", agent.getName(), tool.getName(), tool.getBeanName(), e.getMessage());
                        failedBeans.add(tool.getName() + " (bean'" + tool.getBeanName() + "'不存在)");
                        continue;
                    }
                    
                    if (bean != null) {
                        // 获取原始目标类（处理Spring AOP代理）
                        Class<?> targetClass = getTargetClass(bean);
                        String simpleName = targetClass != null ? targetClass.getSimpleName() : bean.getClass().getSimpleName();
                        
                        toolInstances.add(bean);
                        log.debug("[{}] 成功查找工具'{}' 的Bean实例，原始类: {}", agent.getName(), tool.getName(), simpleName);
                    } else {
                        log.warn("[{}] 工具'{}' 的Bean实例为null", agent.getName(), tool.getName());
                        failedBeans.add(tool.getName() + " (bean实例为null)");
                    }
                } catch (Exception e) {
                    log.error("[{}] 处理工具'{}' 时发生意外错误，详细信息", agent.getName(), tool.getName(), e);
                    failedBeans.add(tool.getName() + " (异常: " + e.getMessage() + ")");
                }
            }
            
            log.info("[{}] 成功获取了{}个工具实例", agent.getName(), toolInstances.size());
            
            // 打印未成功查找的工具（便于故障诊断）
            if (!failedBeans.isEmpty()) {
                log.warn("[{}] 以下工具无法加载: {}", agent.getName(), failedBeans);
            }
            
            return toolInstances;
        } catch (Exception e) {
            log.error("[{}] 获取可用的工具实例时发生了意外错误", agent.getName(), e);
            return List.of();
        }
    }
}