package pangea.hiagent.service;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import pangea.hiagent.model.Agent;
import pangea.hiagent.model.AgentDialogue;
import pangea.hiagent.model.LlmConfig;
import pangea.hiagent.repository.AgentRepository;
import pangea.hiagent.repository.AgentDialogueRepository;
import pangea.hiagent.repository.LlmConfigRepository;
import pangea.hiagent.llm.LlmModelFactory;

import java.util.List;

/**
 * Agent服务类
 * 负责Agent的管理和相关业务逻辑
 */
@Slf4j
@Service
public class AgentService {
    
    private final AgentRepository agentRepository;
    private final AgentDialogueRepository agentDialogueRepository;
    private final LlmConfigRepository llmConfigRepository;
    private final LlmModelFactory llmModelFactory;
    
    public AgentService(AgentRepository agentRepository, 
                       AgentDialogueRepository agentDialogueRepository,
                       LlmConfigRepository llmConfigRepository,
                       LlmModelFactory llmModelFactory) {
        this.agentRepository = agentRepository;
        this.agentDialogueRepository = agentDialogueRepository;
        this.llmConfigRepository = llmConfigRepository;
        this.llmModelFactory = llmModelFactory;
    }
    
    /**
     * 创建Agent
     */
    @Transactional
    public Agent createAgent(Agent agent) {
        log.info("创建Agent: {}", agent.getName());
        
        // 设置默认值
        if (agent.getTemperature() == null) {
            agent.setTemperature(0.7);
        }
        if (agent.getMaxTokens() == null) {
            agent.setMaxTokens(4096);
        }
        if (agent.getTopP() == null) {
            agent.setTopP(0.9);
        }
        if (agent.getTopK() == null) {
            agent.setTopK(50);
        }
        if (agent.getPresencePenalty() == null) {
            agent.setPresencePenalty(0.0);
        }
        if (agent.getFrequencyPenalty() == null) {
            agent.setFrequencyPenalty(0.0);
        }
        if (agent.getHistoryLength() == null) {
            agent.setHistoryLength(10);
        }
        if (agent.getStatus() == null || agent.getStatus().isEmpty()) {
            agent.setStatus("active");
        }
        
        agentRepository.insert(agent);
        return agent;
    }
    
    /**
     * 更新Agent
     */
    @Transactional
    public Agent updateAgent(Agent agent) {
        log.info("更新Agent: {}", agent.getId());
        
        // 保留原始所有者信息
        Agent existingAgent = agentRepository.selectById(agent.getId());
        if (existingAgent != null) {
            agent.setOwner(existingAgent.getOwner());
            agent.setCreatedBy(existingAgent.getCreatedBy());
            agent.setCreatedAt(existingAgent.getCreatedAt());
        }
        
        agentRepository.updateById(agent);
        return agent;
    }
    
    /**
     * 删除Agent
     */
    @Transactional
    public void deleteAgent(String id) {
        log.info("删除Agent: {}", id);
        agentRepository.deleteById(id);
    }
    
    /**
     * 获取Agent详情
     * 
     * @param id Agent ID
     * @return Agent对象，如果不存在则返回null
     */
    public Agent getAgent(String id) {
        if (id == null || id.isEmpty()) {
            log.warn("尝试使用无效ID获取Agent");
            return null;
        }
        return agentRepository.selectById(id);
    }
    
    /**
     * 获取Agent列表
     * 
     * @return Agent列表
     */
    public List<Agent> listAgents() {
        List<Agent> agents = agentRepository.selectList(null);
        log.info("获取到 {} 个Agent", agents != null ? agents.size() : 0);
        return agents != null ? agents : List.of();
    }
    
    /**
     * 分页获取Agent列表
     */
    public IPage<Agent> pageAgents(Long current, Long size, String name, String status) {
        Page<Agent> page = new Page<>(current, size);
        
        LambdaQueryWrapper<Agent> wrapper = new LambdaQueryWrapper<>();
        if (name != null) {
            wrapper.like(Agent::getName, name);
        }
        if (status != null) {
            wrapper.eq(Agent::getStatus, status);
        }
        
        // 使用优化的分页查询方法
        return agentRepository.selectPageWithOptimization(page, wrapper);
    }
    
    /**
     * 获取用户的Agent列表
     */
    public List<Agent> getUserAgents(String userId) {
        // 使用优化的查询方法
        return agentRepository.findActiveAgentsByOwnerWithExplicitColumns(userId);
    }
    
    /**
     * 保存对话记录
     */
    @Transactional
    public void saveDialogue(AgentDialogue dialogue) {
        agentDialogueRepository.insert(dialogue);
    }
    
    /**
     * 根据Agent获取对应的LLM模型
     * 
     * @param agent Agent对象
     * @return 对应的ChatModel实例
     */
    public org.springframework.ai.chat.model.ChatModel getChatModelForAgent(Agent agent) {
        // 获取Agent配置的模型配置名称
        String modelConfigName = agent.getDefaultModel();
        log.info("获取Agent {} 的ChatModel，模型配置名称: {}", agent.getId(), modelConfigName);
        
        if (modelConfigName == null || modelConfigName.isEmpty()) {
            throw new IllegalArgumentException("Agent未配置默认模型");
        }
        
        // 根据模型配置名称查找对应的LLM配置
        LambdaQueryWrapper<LlmConfig> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(LlmConfig::getName, modelConfigName);
        wrapper.eq(LlmConfig::getEnabled, true);
        LlmConfig llmConfig = llmConfigRepository.selectOne(wrapper);
        
        if (llmConfig == null) {
            log.error("未找到启用的LLM配置: {}", modelConfigName);
            throw new IllegalArgumentException("未找到启用的LLM配置: " + modelConfigName);
        }
        
        log.info("找到LLM配置: {} ({})", llmConfig.getName(), llmConfig.getProvider());
        
        // 使用LlmModelFactory创建对应的ChatModel实例
        return llmModelFactory.createChatModel(llmConfig);
    }
}