package pangea.hiagent.security;

import lombok.extern.slf4j.Slf4j;
import org.springframework.security.access.PermissionEvaluator;
import org.springframework.security.core.Authentication;
import org.springframework.stereotype.Component;
import pangea.hiagent.model.Agent;
import pangea.hiagent.service.AgentService;

import java.io.Serializable;

/**
 * 自定义权限评估器
 * 用于实现细粒度的资源级权限控制
 */
@Slf4j
@Component("permissionEvaluator")
public class DefaultPermissionEvaluator implements PermissionEvaluator {
    
    private final AgentService agentService;
    
    public DefaultPermissionEvaluator(AgentService agentService) {
        this.agentService = agentService;
    }
    
    /**
     * 检查用户是否有权访问指定Agent
     */
    @Override
    public boolean hasPermission(Authentication authentication, Object targetDomainObject, Object permission) {
        if (authentication == null || targetDomainObject == null || !(permission instanceof String)) {
            return false;
        }
        
        String userId = (String) authentication.getPrincipal();
        String perm = (String) permission;
        
        // 目前只处理Agent访问权限
        if (targetDomainObject instanceof Agent) {
            Agent agent = (Agent) targetDomainObject;
            return checkAgentAccess(userId, agent, perm);
        } else if (targetDomainObject instanceof String) {
            // 假设targetDomainObject是Agent ID
            String agentId = (String) targetDomainObject;
            Agent agent = agentService.getAgent(agentId);
            if (agent == null) {
                return false;
            }
            return checkAgentAccess(userId, agent, perm);
        }
        
        return false;
    }
    
    @Override
    public boolean hasPermission(Authentication authentication, Serializable targetId, String targetType, Object permission) {
        if (authentication == null || targetId == null || targetType == null || !(permission instanceof String)) {
            return false;
        }
        
        String userId = (String) authentication.getPrincipal();
        String perm = (String) permission;
        
        // 处理基于ID的权限检查
        if ("Agent".equals(targetType)) {
            Agent agent = agentService.getAgent(targetId.toString());
            if (agent == null) {
                return false;
            }
            return checkAgentAccess(userId, agent, perm);
        }
        
        return false;
    }
    
    /**
     * 检查用户对Agent的访问权限
     */
    private boolean checkAgentAccess(String userId, Agent agent, String permission) {
        // 管理员可以访问所有Agent
        if (isAdminUser(userId)) {
            return true;
        }
        
        // 检查Agent所有者
        if (agent.getOwner().equals(userId)) {
            return true;
        }
        
        // 根据权限类型进行检查
        switch (permission.toLowerCase()) {
            case "read":
                // 所有用户都可以读取公开的Agent（如果有此概念）
                return false; // 暂时不支持公开Agent
            case "write":
            case "delete":
            case "execute":
                // 只有所有者可以写入、删除或执行Agent
                return agent.getOwner().equals(userId);
            default:
                return false;
        }
    }
    
    /**
     * 检查是否为管理员用户
     */
    private boolean isAdminUser(String userId) {
        // 这里可以根据实际需求实现管理员检查逻辑
        // 例如查询数据库或检查特殊用户ID
        return "admin".equals(userId) || "user-001".equals(userId);
    }
}