package pangea.hiagent.llm;

import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.chat.model.ChatModel;
import org.springframework.ai.ollama.OllamaChatModel;
import org.springframework.ai.ollama.api.OllamaApi;
import org.springframework.ai.ollama.api.OllamaOptions;
import org.springframework.stereotype.Component;
import pangea.hiagent.model.LlmConfig;

/**
 * Ollama模型适配器
 * 实现Ollama模型的创建和配置
 */
@Slf4j
@Component
public class OllamaModelAdapter implements ModelAdapter {
    
    @Override
    public ChatModel createChatModel(LlmConfig config) {
        if (!validateConfig(config)) {
            throw new IllegalArgumentException("无效的Ollama配置");
        }
        
        try {
            String baseUrl = config.getBaseUrl();
            if (baseUrl == null || baseUrl.isEmpty()) {
                baseUrl = "http://localhost:11434";
            }
            
            // 使用Builder模式创建OllamaChatModel实例
            OllamaApi ollamaApi = new OllamaApi(baseUrl);
            
            // 创建OllamaOptions配置
            OllamaOptions options = OllamaOptions.builder()
                    .model(config.getModelName() != null ? config.getModelName() : "llama2")
                    .temperature(config.getTemperature())
                    .numPredict(config.getMaxTokens())
                    .build();
            
            // 使用Builder模式创建OllamaChatModel实例
            return OllamaChatModel.builder()
                    .ollamaApi(ollamaApi)
                    .defaultOptions(options)
                    .build();
        } catch (Exception e) {
            log.error("创建Ollama ChatModel失败: {}", e.getMessage(), e);
            throw new RuntimeException("创建Ollama ChatModel失败: " + e.getMessage(), e);
        }
    }
    
    @Override
    public String getProviderName() {
        return "ollama";
    }
    
    @Override
    public boolean validateConfig(LlmConfig config) {
        return config != null && config.getEnabled();
    }
}