package pangea.hiagent.agent.service;

import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;
import java.util.Map;

/**
 * 异常监控服务
 * 负责监控、统计和报告系统中的异常情况
 */
@Slf4j
@Service
public class ExceptionMonitoringService {
    
    // 异常统计信息
    private final Map<String, AtomicLong> exceptionCounters = new ConcurrentHashMap<>();
    
    // 异常详细信息缓存
    private final Map<String, String> exceptionDetails = new ConcurrentHashMap<>();
    
    // 最大缓存条目数
    private static final int MAX_CACHE_SIZE = 1000;
    
    /**
     * 记录异常信息
     * 
     * @param exceptionType 异常类型
     * @param errorMessage 错误消息
     * @param stackTrace 堆栈跟踪
     */
    public void recordException(String exceptionType, String errorMessage, String stackTrace) {
        try {
            // 更新异常计数器
            AtomicLong counter = exceptionCounters.computeIfAbsent(exceptionType, k -> new AtomicLong(0));
            counter.incrementAndGet();
            
            // 记录异常详细信息（保留最新的）
            String detailKey = exceptionType + "_" + System.currentTimeMillis();
            exceptionDetails.put(detailKey, formatExceptionDetail(exceptionType, errorMessage, stackTrace));
            
            // 控制缓存大小
            if (exceptionDetails.size() > MAX_CACHE_SIZE) {
                // 移除最老的条目
                String oldestKey = exceptionDetails.keySet().iterator().next();
                exceptionDetails.remove(oldestKey);
            }
            
            // 记录日志
            log.error("异常监控 - 类型: {}, 消息: {}", exceptionType, errorMessage);
        } catch (Exception e) {
            log.warn("记录异常信息时发生错误: {}", e.getMessage());
        }
    }
    
    /**
     * 格式化异常详细信息
     * 
     * @param exceptionType 异常类型
     * @param errorMessage 错误消息
     * @param stackTrace 堆栈跟踪
     * @return 格式化后的异常详细信息
     */
    private String formatExceptionDetail(String exceptionType, String errorMessage, String stackTrace) {
        StringBuilder detail = new StringBuilder();
        detail.append("异常类型: ").append(exceptionType).append("\n");
        detail.append("错误消息: ").append(errorMessage).append("\n");
        detail.append("发生时间: ").append(java.time.Instant.now().toString()).append("\n");
        detail.append("堆栈跟踪:\n").append(stackTrace).append("\n");
        detail.append("---\n");
        return detail.toString();
    }
    
    /**
     * 获取异常统计信息
     * 
     * @return 异常统计信息
     */
    public Map<String, Long> getExceptionStatistics() {
        Map<String, Long> statistics = new ConcurrentHashMap<>();
        for (Map.Entry<String, AtomicLong> entry : exceptionCounters.entrySet()) {
            statistics.put(entry.getKey(), entry.getValue().get());
        }
        return statistics;
    }
    
    /**
     * 获取指定类型的异常计数
     * 
     * @param exceptionType 异常类型
     * @return 异常计数
     */
    public long getExceptionCount(String exceptionType) {
        AtomicLong counter = exceptionCounters.get(exceptionType);
        return counter != null ? counter.get() : 0;
    }
    
    /**
     * 获取所有异常详细信息
     * 
     * @return 异常详细信息
     */
    public Map<String, String> getExceptionDetails() {
        return new ConcurrentHashMap<>(exceptionDetails);
    }
    
    /**
     * 清空异常统计信息
     */
    public void clearStatistics() {
        exceptionCounters.clear();
    }
    
    /**
     * 清空异常详细信息
     */
    public void clearDetails() {
        exceptionDetails.clear();
    }
    
    /**
     * 清空所有异常信息
     */
    public void clearAll() {
        clearStatistics();
        clearDetails();
    }
}