package pangea.hiagent.common.utils;

import io.jsonwebtoken.Claims;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.security.Keys;
import lombok.extern.slf4j.Slf4j;
import pangea.hiagent.common.config.JwtProperties;

import org.springframework.stereotype.Component;

import javax.crypto.SecretKey;
import java.nio.charset.StandardCharsets;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * JWT工具类
 * 负责生成、解析和验证JWT Token
 */
@Slf4j
@Component
public class JwtUtil {
    
    private final JwtProperties jwtProperties;
    
    public JwtUtil(JwtProperties jwtProperties) {
        this.jwtProperties = jwtProperties;
    }
    
    /**
     * 生成JWT Token
     * 
     * @param userId 用户ID
     * @return JWT Token字符串
     */
    public String generateToken(String userId) {
        log.debug("开始生成Token，用户ID: {}", userId);
        if (userId == null || userId.isEmpty()) {
            log.warn("用户ID不能为空");
            return null;
        }
        
        Map<String, Object> claims = new HashMap<>();
        claims.put("userId", userId);
        String token = createToken(claims, userId);
        log.debug("Token生成成功，长度: {}", token != null ? token.length() : 0);
        return token;
    }
    
    /**
     * 生成JWT Token（带自定义claims）
     * 
     * @param userId 用户ID
     * @param claims 自定义claims
     * @return JWT Token字符串
     */
    public String generateToken(String userId, Map<String, Object> claims) {
        log.debug("开始生成Token（带自定义claims），用户ID: {}", userId);
        if (userId == null || userId.isEmpty()) {
            log.warn("用户ID不能为空");
            return null;
        }
        
        if (claims == null) {
            claims = new HashMap<>();
        }
        
        claims.put("userId", userId);
        String token = createToken(claims, userId);
        log.debug("Token生成成功（带自定义claims），长度: {}", token != null ? token.length() : 0);
        return token;
    }
    
    /**
     * 创建Token
     */
    private String createToken(Map<String, Object> claims, String subject) {
        Date now = new Date();
        Date expiration = new Date(now.getTime() + jwtProperties.getExpiration());
        
        log.debug("创建Token - 当前时间: {}, 过期时间: {}, 有效期: {}毫秒", now, expiration, jwtProperties.getExpiration());
        
        return Jwts.builder()
                .claims(claims)
                .subject(subject)
                .issuedAt(now)
                .expiration(expiration)
                .signWith(getSignInKey())
                .compact();
    }
    
    /**
     * 获取签名密钥
     */
    private SecretKey getSignInKey() {
        byte[] keyBytes = jwtProperties.getSecret().getBytes(StandardCharsets.UTF_8);
        return Keys.hmacShaKeyFor(keyBytes);
    }
    
    /**
     * 解析Token获取所有claims
     */
    public Claims getClaimsFromToken(String token) {
        try {
            log.debug("开始解析Token: {}", token);
            Claims claims = Jwts.parser()
                    .verifyWith(getSignInKey())
                    .build()
                    .parseSignedClaims(token)
                    .getPayload();
            log.debug("Token解析成功，claims: {}", claims);
            return claims;
        } catch (Exception e) {
            log.error("获取Token claims失败", e);
            return null;
        }
    }
    
    /**
     * 从Token获取用户ID
     */
    public String getUserIdFromToken(String token) {
        Claims claims = getClaimsFromToken(token);
        if (claims != null) {
            String userId = claims.get("userId", String.class);
            log.debug("从Token中提取用户ID: {}", userId);
            return userId;
        }
        log.warn("无法从Token中提取用户ID，claims为空");
        return null;
    }
    
    /**
     * 从Token获取Subject
     */
    public String getSubjectFromToken(String token) {
        Claims claims = getClaimsFromToken(token);
        if (claims != null) {
            String subject = claims.getSubject();
            log.debug("从Token中提取Subject: {}", subject);
            return subject;
        }
        log.warn("无法从Token中提取Subject，claims为空");
        return null;
    }
    
    /**
     * 检查Token是否过期
     */
    public Boolean isTokenExpired(String token) {
        try {
            Claims claims = getClaimsFromToken(token);
            if (claims == null) {
                log.debug("Token claims为空，标记为过期");
                return true;
            }
            Date expiration = claims.getExpiration();
            boolean expired = expiration.before(new Date());
            log.debug("Token过期检查结果: {}，过期时间: {}, 当前时间: {}", expired, expiration, new Date());
            return expired;
        } catch (Exception e) {
            log.error("检查Token是否过期时发生异常", e);
            return true; // 出现异常时认为token已过期
        }
    }
    
    /**
     * 验证Token是否有效
     */
    public Boolean validateToken(String token, String userId) {
        String tokenUserId = getUserIdFromToken(token);
        boolean isValid = tokenUserId != null && tokenUserId.equals(userId) && !isTokenExpired(token);
        log.debug("Token验证结果: {}, Token用户ID: {}, 期望用户ID: {}, 是否过期: {}", 
            isValid, tokenUserId, userId, isTokenExpired(token));
        return isValid;
    }
    
    /**
     * 验证Token是否有效（仅检查过期时间）
     */
    public Boolean validateToken(String token) {
        try {
            log.debug("开始验证Token: {}", token);
            boolean isValid = !isTokenExpired(token);
            log.debug("Token验证结果: {}, 是否过期: {}", isValid, isTokenExpired(token));
            return isValid;
        } catch (Exception e) {
            log.error("验证Token时发生异常", e);
            return false; // 出现异常时认为token无效
        }
    }
}