package pangea.hiagent.workpanel.playwright;

import com.microsoft.playwright.*;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;
import org.springframework.context.annotation.Lazy;

import jakarta.annotation.PostConstruct;
import jakarta.annotation.PreDestroy;
import java.util.concurrent.*;

/**
 * Playwright管理器实现类
 * 负责统一管理Playwright实例和用户隔离的BrowserContext
 */
@Slf4j
@Component
@Lazy
public class PlaywrightManagerImpl implements PlaywrightManager {
    
    // 共享的Playwright实例
    private volatile Playwright playwright;
    
    // 共享的浏览器实例
    private volatile Browser browser;
    
    // 用户浏览器上下文映射表（用户ID -> BrowserContext）
    private final ConcurrentMap<String, BrowserContext> userContexts = new ConcurrentHashMap<>();
    
    // 用户上下文创建时间映射表（用于超时清理）
    private final ConcurrentMap<String, Long> contextCreationTimes = new ConcurrentHashMap<>();
    
    // 用户上下文超时时间（毫秒），默认30分钟
    private static final long CONTEXT_TIMEOUT = 30 * 60 * 1000;
    
    // 清理任务调度器
    private ScheduledExecutorService cleanupScheduler;
    
    // 标记是否已经初始化
    private volatile boolean initialized = false;
    
    // 用于同步初始化过程
    private final Object initLock = new Object();
    
    /**
     * 延迟初始化Playwright和浏览器实例
     */
    private void lazyInitialize() {
        if (!initialized) {
            synchronized (initLock) {
                if (!initialized) {
                    try {
                        log.info("正在初始化Playwright管理器...");
                        
                        // 创建Playwright实例
                        this.playwright = Playwright.create();
                        
                        // 启动Chrome浏览器，无头模式
                        this.browser = playwright.chromium().launch(new BrowserType.LaunchOptions()
                                .setHeadless(true)
                                .setArgs(java.util.Arrays.asList(
                                        "--no-sandbox",
                                        "--disable-dev-shm-usage",
                                        "--disable-gpu",
                                        "--remote-allow-origins=*")));
                        
                        // 初始化清理任务调度器
                        this.cleanupScheduler = Executors.newSingleThreadScheduledExecutor();
                        
                        // 每5分钟检查一次超时的用户上下文
                        this.cleanupScheduler.scheduleAtFixedRate(this::cleanupExpiredContexts, 
                                5, 5, TimeUnit.MINUTES);
                        
                        this.initialized = true;
                        log.info("Playwright管理器初始化成功");
                    } catch (Exception e) {
                        log.error("Playwright管理器初始化失败: ", e);
                        throw new RuntimeException("Failed to initialize Playwright manager", e);
                    }
                }
            }
        }
    }
    
    // 移除@PostConstruct注解，避免在Spring初始化时自动调用
    /*
    @PostConstruct
    public void initialize() {
        lazyInitialize();
    }
    */
    
    @Override
    public Playwright getPlaywright() {
        lazyInitialize();
        if (playwright == null) {
            throw new IllegalStateException("Playwright instance is not initialized");
        }
        return playwright;
    }
    
    @Override
    public Browser getBrowser() {
        lazyInitialize();
        if (browser == null || !browser.isConnected()) {
            throw new IllegalStateException("Browser instance is not available");
        }
        return browser;
    }
    
    @Override
    public BrowserContext getUserContext(String userId) {
        lazyInitialize();
        Browser.NewContextOptions options = new Browser.NewContextOptions()
                  .setViewportSize(1344, 2992) // 设置视口大小，与前端一致;手机型号：Google Pixel 9 Pro XL
                  .setUserAgent("Mozilla/5.0 (Linux; Android 15; Pixel 9 Pro XL Build/UP2A.250105.004) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/129.0.0.0 Mobile Safari/537.36"); // 设置用户代理
        return getUserContext(userId, options);
    }
    
    @Override
    public BrowserContext getUserContext(String userId, Browser.NewContextOptions options) {
        lazyInitialize();
        if (userId == null || userId.isEmpty()) {
            throw new IllegalArgumentException("User ID cannot be null or empty");
        }
        
        if (options == null) {
            options = new Browser.NewContextOptions();
        }
        
        // 尝试从缓存中获取已存在的上下文
        BrowserContext context = userContexts.get(userId);
        
        // 如果上下文不存在或已关闭，则创建新的
        if (context == null || context.pages().isEmpty()) {
            try {
                log.debug("为用户 {} 创建新的浏览器上下文", userId);
                context = browser.newContext(options);
                userContexts.put(userId, context);
                contextCreationTimes.put(userId, System.currentTimeMillis());
            } catch (Exception e) {
                log.error("为用户 {} 创建浏览器上下文失败", userId, e);
                throw new RuntimeException("Failed to create browser context for user: " + userId, e);
            }
        }
        
        return context;
    }
    
    @Override
    public void releaseUserContext(String userId) {
        if (userId == null || userId.isEmpty()) {
            return;
        }
        
        BrowserContext context = userContexts.remove(userId);
        contextCreationTimes.remove(userId);
        
        if (context != null) {
            try {
                context.close();
                log.debug("用户 {} 的浏览器上下文已释放", userId);
            } catch (Exception e) {
                log.warn("关闭用户 {} 的浏览器上下文时发生异常", userId, e);
            }
        }
    }
    
    /**
     * 清理过期的用户上下文
     */
    private void cleanupExpiredContexts() {
        long currentTime = System.currentTimeMillis();
        long expiredThreshold = currentTime - CONTEXT_TIMEOUT;
        
        for (String userId : contextCreationTimes.keySet()) {
            Long creationTime = contextCreationTimes.get(userId);
            if (creationTime != null && creationTime < expiredThreshold) {
                log.info("清理过期的用户上下文: {}", userId);
                releaseUserContext(userId);
            }
        }
    }
    
    /**
     * 销毁所有资源
     */
    @PreDestroy
    @Override
    public void destroy() {
        log.info("开始销毁Playwright管理器资源...");
        
        try {
            // 关闭清理任务调度器
            if (cleanupScheduler != null) {
                cleanupScheduler.shutdown();
                if (!cleanupScheduler.awaitTermination(5, TimeUnit.SECONDS)) {
                    cleanupScheduler.shutdownNow();
                }
            }
        } catch (Exception e) {
            log.warn("关闭清理任务调度器时发生异常", e);
        }
        
        // 关闭所有用户上下文
        for (String userId : userContexts.keySet()) {
            releaseUserContext(userId);
        }
        
        // 关闭浏览器
        try {
            if (browser != null && browser.isConnected()) {
                browser.close();
                log.info("浏览器实例已关闭");
            }
        } catch (Exception e) {
            log.warn("关闭浏览器实例时发生异常", e);
        }
        
        // 关闭Playwright
        try {
            if (playwright != null) {
                playwright.close();
                log.info("Playwright实例已关闭");
            }
        } catch (Exception e) {
            log.warn("关闭Playwright实例时发生异常", e);
        }
        
        log.info("Playwright管理器资源已全部销毁");
    }
}