package pangea.hiagent.security;

import lombok.extern.slf4j.Slf4j;
import org.springframework.security.access.PermissionEvaluator;
import org.springframework.security.core.Authentication;
import org.springframework.stereotype.Component;
import pangea.hiagent.model.Agent;
import pangea.hiagent.model.TimerConfig;
import pangea.hiagent.web.service.AgentService;
import pangea.hiagent.web.service.TimerService;

import java.io.Serializable;

/**
 * 自定义权限评估器
 * 用于实现细粒度的资源级权限控制
 */
@Slf4j
@Component("permissionEvaluator")
public class DefaultPermissionEvaluator implements PermissionEvaluator {
    
    private static final String AGENT_TYPE = "Agent";
    private static final String TIMER_CONFIG_TYPE = "TimerConfig";
    
    private final AgentService agentService;
    private final TimerService timerService;
    
    public DefaultPermissionEvaluator(AgentService agentService, TimerService timerService) {
        this.agentService = agentService;
        this.timerService = timerService;
    }
    
    /**
     * 检查用户是否有权访问指定资源
     */
    @Override
    public boolean hasPermission(Authentication authentication, Object targetDomainObject, Object permission) {
        if (authentication == null || targetDomainObject == null || !(permission instanceof String)) {
            return false;
        }
        
        String userId = authentication.getPrincipal().toString();
        String perm = (String) permission;
        
        try {
            // 处理Agent访问权限
            if (targetDomainObject instanceof Agent) {
                return checkAgentAccess(userId, (Agent) targetDomainObject, perm);
            } 
            // 处理TimerConfig访问权限
            else if (targetDomainObject instanceof TimerConfig) {
                return checkTimerAccess(userId, (TimerConfig) targetDomainObject, perm);
            }
        } catch (Exception e) {
            log.error("权限检查异常: userId={}, target={}, permission={}, error={}", 
                    userId, targetDomainObject.getClass().getSimpleName(), perm, e.getMessage());
        }
        
        return false;
    }
    
    @Override
    public boolean hasPermission(Authentication authentication, Serializable targetId, String targetType, Object permission) {
        if (authentication == null || targetId == null || targetType == null || !(permission instanceof String)) {
            return false;
        }
        
        String userId = authentication.getPrincipal().toString();
        String perm = (String) permission;
        
        try {
            // 处理基于ID的权限检查
            if (AGENT_TYPE.equals(targetType)) {
                Agent agent = agentService.getAgent(targetId.toString());
                return agent != null && checkAgentAccess(userId, agent, perm);
            } 
            // 处理TimerConfig资源的权限检查
            else if (TIMER_CONFIG_TYPE.equals(targetType)) {
                TimerConfig timer = timerService.getTimerById(targetId.toString());
                return timer != null && checkTimerAccess(userId, timer, perm);
            }
        } catch (Exception e) {
            log.error("基于ID的权限检查异常: userId={}, targetId={}, targetType={}, permission={}, error={}", 
                    userId, targetId, targetType, perm, e.getMessage());
        }
        
        return false;
    }
    
    /**
     * 检查用户对Agent的访问权限
     */
    private boolean checkAgentAccess(String userId, Agent agent, String permission) {
        // 管理员可以访问所有Agent
        if (isAdminUser(userId)) {
            return true;
        }
        
        // 所有者可以访问
        if (agent.getOwner().equals(userId)) {
            return true;
        }
        
        // 根据权限类型进行检查（目前只支持所有者访问）
        String permissionLower = permission.toLowerCase();
        return switch (permissionLower) {
            case "read", "write", "delete", "execute" -> agent.getOwner().equals(userId);
            default -> false;
        };
    }
    
    /**
     * 检查用户对TimerConfig的访问权限
     */
    private boolean checkTimerAccess(String userId, TimerConfig timer, String permission) {
        // 管理员可以访问所有定时器
        if (isAdminUser(userId)) {
            return true;
        }
        
        // 创建者可以访问
        if (timer.getCreatedBy() != null && timer.getCreatedBy().equals(userId)) {
            return true;
        }
        
        // 根据权限类型进行检查（目前只支持创建者访问）
        String permissionLower = permission.toLowerCase();
        return switch (permissionLower) {
            case "read", "write", "delete" -> timer.getCreatedBy() != null && timer.getCreatedBy().equals(userId);
            default -> false;
        };
    }
    
    /**
     * 检查是否为管理员用户
     */
    private boolean isAdminUser(String userId) {
        // 管理员用户检查，可扩展为从配置或数据库读取
        return "admin".equals(userId) || "user-001".equals(userId);
    }
}