# OAuth2 快速配置指南

## 一、系统要求

- Java 17+
- Spring Boot 3.x
- MySQL 8.0+ 或 H2 数据库
- Redis（可选，用于缓存）

## 二、配置 OAuth2 提供者（以 GitHub 为例）

### 2.1 获取 OAuth2 凭证

访问 GitHub Developer Settings：
1. 登录 GitHub 账户
2. 进入 Settings → Developer settings → OAuth Apps
3. 点击 "New OAuth App"
4. 填写应用信息：
   - **Application name**: HiAgent
   - **Homepage URL**: http://localhost:8081
   - **Authorization callback URL**: http://localhost:8081/api/v1/auth/oauth2/callback?providerName=github
5. 获取 **Client ID** 和 **Client Secret**

### 2.2 配置数据库

在数据库中插入 OAuth2 提供者配置：

```sql
INSERT INTO oauth2_provider (
  id, 
  provider_name, 
  display_name, 
  description,
  auth_type,
  authorize_url, 
  token_url, 
  userinfo_url,
  client_id, 
  client_secret, 
  redirect_uri, 
  scope,
  enabled
) VALUES (
  UUID(),
  'github',
  'GitHub 登录',
  'GitHub OAuth2 认证',
  'authorization_code',
  'https://github.com/login/oauth/authorize',
  'https://github.com/login/oauth/access_token',
  'https://api.github.com/user',
  'YOUR_CLIENT_ID_HERE',
  'YOUR_CLIENT_SECRET_HERE',
  'http://localhost:8081/api/v1/auth/oauth2/callback?providerName=github',
  'user:email',
  1
);
```

### 2.3 配置前端

在 Login.vue 中，OAuth2 提供者会自动显示在登录页面。

## 三、OAuth2 认证流程

### 3.1 用户点击 "GitHub 登录"

```
用户 → 前端 Login.vue
              ↓
        点击 GitHub 按钮
              ↓
        window.location.href = '/api/v1/auth/oauth2/authorize?providerName=github'
```

### 3.2 后端授权端点处理

```
GET /api/v1/auth/oauth2/authorize?providerName=github
              ↓
        获取 GitHub 配置
              ↓
        构造授权 URL（包含 client_id, redirect_uri, scope）
              ↓
        重定向到 GitHub 授权服务器
              ↓
GitHub: https://github.com/login/oauth/authorize?client_id=xxx&redirect_uri=...
```

### 3.3 用户在 GitHub 上授权

用户在 GitHub 上验证身份并授权应用访问用户信息。

### 3.4 GitHub 回调

```
GitHub 重定向到：
http://localhost:8081/api/v1/auth/oauth2/callback?
  code=AUTH_CODE&
  state=STATE&
  providerName=github
              ↓
        后端处理回调请求
```

### 3.5 后端令牌交换

```
后端收到授权码后：
              ↓
1. 使用授权码向 GitHub 请求访问令牌
   POST https://github.com/login/oauth/access_token
   Content-Type: application/x-www-form-urlencoded
   code=AUTH_CODE&
   client_id=CLIENT_ID&
   client_secret=CLIENT_SECRET&
   redirect_uri=REDIRECT_URI
              ↓
2. 使用访问令牌获取用户信息
   GET https://api.github.com/user
   Authorization: Bearer ACCESS_TOKEN
              ↓
3. 查找或创建用户
   ├─ 检查是否存在 oauth2_account 关联
   ├─ 检查是否存在同邮箱的用户
   └─ 否则创建新用户
              ↓
4. 保存 OAuth2 账户关联
   INSERT INTO oauth2_account ...
              ↓
5. 生成 JWT Token
   token = jwtUtil.generateToken(user.getId())
              ↓
6. 重定向到前端
   redirect('/login?token=JWT_TOKEN&method=oauth2')
```

### 3.6 前端处理回调

```
onMounted() {
  检查 URL 参数
  ├─ 发现 token 和 method=oauth2
  ├─ 保存 token 到 localStorage
  ├─ 更新 authStore
  └─ 重定向到 /agent/chat
}
```

## 四、配置其他 OAuth2 提供者

### 4.1 Google 登录

1. **获取凭证**：
   - 访问 [Google Cloud Console](https://console.cloud.google.com/)
   - 创建 OAuth2 凭证

2. **配置数据库**：
```sql
INSERT INTO oauth2_provider (
  id, provider_name, display_name, auth_type,
  authorize_url, token_url, userinfo_url,
  client_id, client_secret, redirect_uri, scope, enabled
) VALUES (
  UUID(), 'google', 'Google 登录', 'authorization_code',
  'https://accounts.google.com/o/oauth2/v2/auth',
  'https://oauth2.googleapis.com/token',
  'https://www.googleapis.com/oauth2/v2/userinfo',
  'YOUR_CLIENT_ID', 'YOUR_CLIENT_SECRET',
  'http://localhost:8081/api/v1/auth/oauth2/callback?providerName=google',
  'openid email profile', 1
);
```

### 4.2 Microsoft 登录

类似配置，使用 Microsoft Azure AD 的端点：
- authorize_url: `https://login.microsoftonline.com/common/oauth2/v2.0/authorize`
- token_url: `https://login.microsoftonline.com/common/oauth2/v2.0/token`
- userinfo_url: `https://graph.microsoft.com/v1.0/me`

## 五、测试 OAuth2 流程

### 5.1 本地测试环境

确保使用正确的回调 URL：
```
http://localhost:8081/api/v1/auth/oauth2/callback?providerName=github
```

### 5.2 测试步骤

1. 启动后端：`mvn spring-boot:run`
2. 启动前端：`npm run dev`
3. 访问 http://localhost:5173/login
4. 点击 "GitHub 登录" 按钮
5. 在 GitHub 上授权应用
6. 应自动重定向到 /agent/chat

## 六、故障排查

### 常见错误

| 错误 | 原因 | 解决方案 |
|------|------|--------|
| "未找到配置的 OAuth2 提供者" | 数据库中没有提供者配置 | 检查 oauth2_provider 表 |
| "令牌交换失败" | client_id/secret 错误或网络问题 | 检查凭证和日志 |
| "获取用户信息失败" | 访问令牌无效或 API 端点错误 | 检查 userinfo_url 和作用域 |
| "用户创建失败" | 用户名重复 | 检查 sys_user 表中的 username 唯一性 |

### 查看日志

所有错误都会记录详细的堆栈信息：
```bash
# 查看 ERROR 级别的日志
grep ERROR logs/hiagent.log

# 查看认证相关的日志
grep "OAuth2\|认证\|登录" logs/hiagent.log
```

## 七、安全建议

1. **保护 Client Secret**
   - 不要在前端代码中暴露 client_secret
   - 只在后端服务器上存储

2. **HTTPS**
   - 生产环境必须使用 HTTPS
   - 确保 redirect_uri 使用 HTTPS

3. **State 参数**
   - 系统自动生成 state 参数防止 CSRF
   - 用户不需要手动配置

4. **Scope 权限**
   - 只请求必要的权限
   - 定期审查和更新权限列表

## 八、参考资源

- [GitHub OAuth 文档](https://docs.github.com/en/developers/apps/building-oauth-apps)
- [Google OAuth 文档](https://developers.google.com/identity/protocols/oauth2)
- [OAuth2 标准](https://tools.ietf.org/html/rfc6749)

## 九、常见问题

**Q: 用户已使用 OAuth2 登录，现在想用用户名/密码登录？**
A: 系统支持同一用户使用多种认证方式。用户邮箱相同时会自动关联。

**Q: 如何撤销 OAuth2 账户关联？**
A: 手动删除 oauth2_account 表中的相应记录。

**Q: 能否强制用户只使用某种认证方式？**
A: 可以。在 AuthConfigService 中禁用不需要的认证模式。

---

**最后更新**：2024-12-15  
**文档版本**：1.0
