# 多种登录模式统一认证架构设计文档

## 一、系统概述

本文档详细描述了 HiAgent 系统中实现的**统一身份认证架构**，该架构支持多种登录模式，包括：
1. **本地用户名/密码认证**（LOCAL）
2. **OAuth2.0 授权码模式**（OAUTH2_AUTHORIZATION_CODE）
3. **支持未来扩展**：LDAP、SAML 等其他认证机制

## 二、系统架构

### 2.1 整体架构图

```
┌─────────────────────────────────────────────────────────────┐
│                      前端应用 (Vue3)                         │
│  ┌─────────────────────────────────────────────────────────┐
│  │ Login.vue                                               │
│  │ ├── 本地登录表单                                         │
│  │ └── OAuth2 登录选项                                      │
│  └─────────────────────────────────────────────────────────┘
└─────────────────────────────────────────────────────────────┘
              ↓                          ↓
    /api/v1/auth/login      /api/v1/auth/oauth2/authorize
              ↓                          ↓
┌─────────────────────────────────────────────────────────────┐
│                   认证控制器 (AuthController)                │
│  ├── POST /login                                            │
│  ├── GET /oauth2/authorize                                  │
│  ├── GET /oauth2/callback                                   │
│  └── POST /oauth2/token                                     │
└─────────────────────────────────────────────────────────────┘
              ↓
┌─────────────────────────────────────────────────────────────┐
│              统一认证门面 (UnifiedAuthService)              │
│  ├── authenticate(authMode, credentials)                   │
│  ├── loginWithLocal(username, password)                    │
│  ├── loginWithOAuth2(code, providerName)                   │
│  └── verifyToken(token)                                     │
└─────────────────────────────────────────────────────────────┘
              ↓
┌─────────────────────────────────────────────────────────────┐
│           认证策略实现 (AuthenticationStrategy)             │
│  ├── LocalAuthenticationStrategy                           │
│  └── OAuth2AuthenticationStrategy                          │
└─────────────────────────────────────────────────────────────┘
              ↓
┌─────────────────────────────────────────────────────────────┐
│                    数据持久层 (Repository)                   │
│  ├── UserRepository                                         │
│  ├── OAuth2ProviderRepository                              │
│  └── OAuth2AccountRepository                               │
└─────────────────────────────────────────────────────────────┘
```

### 2.2 核心组件说明

#### 2.2.1 认证模式枚举 (AuthMode.java)
定义系统支持的所有认证方式：
- `LOCAL`: 本地用户名/密码认证
- `OAUTH2_AUTHORIZATION_CODE`: OAuth2.0 授权码模式
- `OAUTH2_IMPLICIT`: OAuth2.0 隐式授权（预留）
- `OAUTH2_CLIENT_CREDENTIALS`: OAuth2.0 客户端凭证（预留）
- `LDAP`: LDAP 目录认证（预留）
- `SAML`: SAML 单点登录（预留）

#### 2.2.2 认证策略接口 (AuthenticationStrategy.java)
定义统一的认证流程接口：
```java
public interface AuthenticationStrategy {
    String getName();
    boolean supports(String authMode);
    String authenticate(Map<String, Object> credentials);
    String refreshToken(String refreshToken);
    boolean verify(String token);
}
```

#### 2.2.3 本地认证策略 (LocalAuthenticationStrategy.java)
实现基于本地数据库的用户名/密码认证：
- 查询用户信息
- 验证密码（支持开发环境跳过）
- 检查用户状态
- 生成 JWT Token
- **包含完整的错误日志输出**

#### 2.2.4 OAuth2 认证策略 (OAuth2AuthenticationStrategy.java)
实现标准 OAuth2.0 授权码流程：
- **授权码交换**：使用授权码交换访问令牌
- **用户信息获取**：从 OAuth2 提供者获取用户信息
- **用户创建/查找**：自动创建新用户或关联现有用户
- **账户关联**：保存 OAuth2 账户关联信息
- **令牌管理**：保存访问令牌和刷新令牌
- **包含详细的错误堆栈追踪**

#### 2.2.5 统一认证门面 (UnifiedAuthService.java)
协调所有认证策略的执行：
```java
public String authenticate(String authMode, Map<String, Object> credentials)
public String loginWithLocal(String username, String password)
public String loginWithOAuth2(String authorizationCode, String providerName)
public boolean verifyToken(String token)
public String buildOAuth2AuthorizationUrl(String providerName)
public Map<String, Object> getAvailableAuthModes()
```

#### 2.2.6 认证配置管理 (AuthConfigService.java)
管理各种认证模式的启用/禁用状态和配置信息：
- `isAuthModeEnabled(authMode)`: 检查认证模式是否启用
- `enableAuthMode(authMode)`: 启用认证模式
- `disableAuthMode(authMode)`: 禁用认证模式
- `setAuthModeConfig(authMode, config)`: 设置认证模式配置

## 三、数据库架构

### 3.1 新增数据库表

#### 3.1.1 oauth2_provider 表
存储 OAuth2 提供者的配置信息：
```sql
CREATE TABLE oauth2_provider (
  id VARCHAR(36) PRIMARY KEY,
  provider_name VARCHAR(50) UNIQUE,
  display_name VARCHAR(100),
  auth_type VARCHAR(50),
  authorize_url VARCHAR(255),
  token_url VARCHAR(255),
  userinfo_url VARCHAR(255),
  client_id VARCHAR(255),
  client_secret VARCHAR(255),
  redirect_uri VARCHAR(255),
  scope VARCHAR(255),
  enabled TINYINT DEFAULT 1,
  config_json JSON,
  ...时间戳字段...
);
```

#### 3.1.2 oauth2_account 表
记录用户与 OAuth2 提供者的账户关联：
```sql
CREATE TABLE oauth2_account (
  id VARCHAR(36) PRIMARY KEY,
  user_id VARCHAR(36),
  provider_name VARCHAR(50),
  remote_user_id VARCHAR(255),
  remote_username VARCHAR(100),
  remote_email VARCHAR(100),
  access_token VARCHAR(1000),
  refresh_token VARCHAR(1000),
  token_expiry TIMESTAMP,
  scope VARCHAR(255),
  profile_data JSON,
  linked_at TIMESTAMP,
  last_login_at TIMESTAMP,
  FOREIGN KEY (user_id) REFERENCES sys_user(id)
);
```

#### 3.1.3 login_mode_config 表
存储登录模式配置（用于未来扩展）：
```sql
CREATE TABLE login_mode_config (
  id VARCHAR(36) PRIMARY KEY,
  mode_name VARCHAR(50) UNIQUE,
  mode_type VARCHAR(50),
  display_name VARCHAR(100),
  enabled TINYINT DEFAULT 1,
  config_json JSON,
  display_order INT DEFAULT 0,
  ...时间戳字段...
);
```

## 四、API 端点设计

### 4.1 本地登录（保持不变）

**请求：**
```bash
POST /api/v1/auth/login
Content-Type: application/json

{
  "username": "user123",
  "password": "password123"
}
```

**响应：**
```json
{
  "code": 0,
  "message": "登录成功",
  "data": {
    "token": "eyJhbGciOiJIUzI1NiIs...",
    "expiresIn": 7200,
    "tokenType": "Bearer"
  }
}
```

### 4.2 OAuth2 授权端点

**请求：**
```bash
GET /api/v1/auth/oauth2/authorize?providerName=github
```

**处理流程：**
1. 后端获取 OAuth2 提供者配置
2. 构造授权 URL（包含 client_id、redirect_uri、scope 等）
3. **重定向用户到授权服务器**
4. 用户在授权服务器上进行身份验证
5. 授权服务器重定向到 redirect_uri（回调端点）

### 4.3 OAuth2 回调端点

**请求：**
```bash
GET /api/v1/auth/oauth2/callback?code=AUTH_CODE&state=STATE&providerName=github
```

**处理流程：**
1. 验证 state 参数（防 CSRF）
2. 使用授权码交换访问令牌
3. 从授权服务器获取用户信息
4. 自动创建或关联用户
5. 保存 OAuth2 账户信息
6. **重定向到前端**，传递 JWT Token

### 4.4 OAuth2 令牌交换端点（后端调用）

**请求：**
```bash
POST /api/v1/auth/oauth2/token
Content-Type: application/json

{
  "authorizationCode": "AUTH_CODE",
  "providerName": "github"
}
```

**响应：**
```json
{
  "code": 0,
  "message": "OAuth2 认证成功",
  "data": {
    "token": "eyJhbGciOiJIUzI1NiIs...",
    "expiresIn": 7200,
    "tokenType": "Bearer"
  }
}
```

## 五、前端集成

### 5.1 登录页面改进

**支持的登录方式：**
1. 用户名/密码登录
2. OAuth2 授权（如 GitHub）
3. 支持显示可用的第三方登录选项

### 5.2 OAuth2 流程处理

**前端 auth.ts 中的新函数：**
```typescript
// 发起 OAuth2 授权
async function loginWithOAuth2(providerName: string)

// 处理 OAuth2 回调
function handleOAuth2Callback(): Promise<any>

// 使用授权码交换令牌
async function loginWithOAuth2Code(authCode: string, providerName: string)
```

### 5.3 回调处理

前端自动检测 URL 参数：
- `token`: JWT 令牌
- `method=oauth2`: 指示认证方式
- `error`: 错误信息

## 六、错误处理和日志

### 6.1 异常日志输出

所有异常处理中都包含详细的错误堆栈信息：

```java
// 示例
log.error("令牌交换失败: 提供者={}, statusCode={}, responseBody={}, 错误堆栈: ", 
         provider.getProviderName(), response.getStatusCode(), response.getBody(), e);
```

### 6.2 错误类型

- `RuntimeException`: 认证失败、用户不存在、密码错误等
- 具体错误消息会返回给前端用于用户反馈

## 七、安全考虑

### 7.1 密码安全
- 使用 BCryptPasswordEncoder 进行密码加密
- 生产环境不记录密码相关信息
- 开发环境支持跳过密码验证（便于测试）

### 7.2 令牌安全
- 使用 JWT Token（JWT 签名确保完整性）
- 令牌过期时间为 2 小时
- 刷新令牌过期时间为 7 天

### 7.3 OAuth2 安全
- 使用 state 参数防止 CSRF 攻击
- 客户端密钥存储在后端（不暴露给前端）
- 访问令牌存储在数据库中
- 刷新令牌存储在数据库中（用于令牌续期）

### 7.4 数据库安全
- OAuth2 账户与用户账户通过外键关联
- 逻辑删除标志防止数据硬删除
- 使用唯一约束防止重复关联

## 八、可扩展性设计

### 8.1 添加新的认证方式

要添加新的认证方式（如 LDAP），只需：

1. **创建新的策略类**
```java
@Component
public class LdapAuthenticationStrategy implements AuthenticationStrategy {
    @Override
    public boolean supports(String authMode) {
        return AuthMode.LDAP.getCode().equals(authMode);
    }
    
    @Override
    public String authenticate(Map<String, Object> credentials) {
        // 实现 LDAP 认证逻辑
    }
}
```

2. **注册到 Spring 容器**
   - Spring 会自动注入到 UnifiedAuthService 的 List<AuthenticationStrategy>

3. **无需修改其他代码**
   - 统一的接口确保兼容性

### 8.2 添加新的 OAuth2 提供者

只需在 `oauth2_provider` 表中插入配置记录：

```sql
INSERT INTO oauth2_provider (
  id, provider_name, display_name, auth_type,
  authorize_url, token_url, userinfo_url,
  client_id, client_secret, redirect_uri, scope, enabled
) VALUES (
  'uuid', 'google', 'Google 登录', 'authorization_code',
  'https://accounts.google.com/o/oauth2/v2/auth',
  'https://oauth2.googleapis.com/token',
  'https://www.googleapis.com/oauth2/v2/userinfo',
  'client_id', 'client_secret', 'http://localhost:5173/login', 'openid email profile', 1
);
```

## 九、测试

### 9.1 单元测试
- LocalAuthenticationStrategy 测试
- OAuth2AuthenticationStrategy 测试（使用 Mock）
- AuthConfigService 测试

### 9.2 集成测试
- UnifiedAuthServiceIntegrationTest
  - 测试本地认证流程
  - 测试错误处理
  - 测试认证模式启用/禁用
  - 测试令牌验证

## 十、部署建议

### 10.1 环境配置

**application.yml 配置示例：**
```yaml
hiagent:
  jwt:
    secret: ${JWT_SECRET:your-secret-key}
    expiration: 7200000
```

### 10.2 数据库初始化

运行 schema.sql 初始化数据库表。

### 10.3 OAuth2 提供者配置

在应用启动前，在 `oauth2_provider` 表中配置需要的 OAuth2 提供者。

## 十一、总结

本架构设计实现了一个**灵活、可扩展、易于维护**的多种认证方式统一管理系统：

✓ **支持多种认证模式**：本地认证、OAuth2、预留 LDAP、SAML 等
✓ **易于扩展**：新增认证方式无需修改现有代码
✓ **向后兼容**：原有的登录接口保持不变
✓ **完整的错误日志**：便于问题定位和系统维护
✓ **安全可靠**：加密存储、令牌管理、CSRF 防护等
✓ **生产就绪**：包含完整的测试和文档
