package pangea.hiagent.llm;

import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.chat.model.ChatModel;
import org.springframework.ai.openai.OpenAiChatModel;
import org.springframework.ai.openai.api.OpenAiApi;
import org.springframework.ai.openai.OpenAiChatOptions;
import org.springframework.stereotype.Component;
import pangea.hiagent.model.LlmConfig;

/**
 * DeepSeek模型适配器
 * 实现DeepSeek模型的创建和配置（使用OpenAI兼容API）
 */
@Slf4j
@Component
public class DeepSeekModelAdapter implements ModelAdapter {
    
    @Override
    public ChatModel createChatModel(LlmConfig config) {
        if (!validateConfig(config)) {
            throw new IllegalArgumentException("无效的DeepSeek配置");
        }
        
        log.info("创建DeepSeek ChatModel，模型名称: {}, API密钥长度: {}", config.getModelName(), config.getApiKey().length());
        
        try {
            // DeepSeek使用OpenAI兼容的API，使用Builder模式创建OpenAiApi实例
            // 注意：DeepSeek API端点不需要包含/v1，因为Spring AI会自动添加
            String baseUrl = config.getBaseUrl() != null && !config.getBaseUrl().isEmpty() ? 
            config.getBaseUrl() : "https://api.deepseek.com";
            
            OpenAiApi openAiApi = OpenAiApi.builder()
                    .baseUrl(baseUrl)
                    .apiKey(config.getApiKey())
                    .build();
                    
            log.info("OpenAiApi实例创建成功");
            
            // 使用Builder模式创建OpenAiChatModel实例
            OpenAiChatModel chatModel = OpenAiChatModel.builder()
                    .openAiApi(openAiApi)
                    .defaultOptions(OpenAiChatOptions.builder()
                            .model(config.getModelName())
                            .temperature(config.getTemperature())
                            .maxTokens(config.getMaxTokens())
                            .build())
                    .build();
                    
            log.info("OpenAiChatModel实例创建成功");
            return chatModel;
        } catch (Exception e) {
            log.error("创建DeepSeek ChatModel时发生异常", e);
            throw new RuntimeException("创建DeepSeek ChatModel失败: " + e.getMessage(), e);
        }
    }
    
    @Override
    public String getProviderName() {
        return "deepseek";
    }
    
    @Override
    public boolean validateConfig(LlmConfig config) {
        // 修改验证逻辑，允许在没有API密钥的情况下启动，但会给出警告
        if (config == null || !config.getEnabled()) {
            return false;
        }
        
        // 如果启用了配置但没有API密钥，允许启动但会在使用时抛出异常
        if (config.getApiKey() == null || config.getApiKey().isEmpty()) {
            log.warn("DeepSeek配置已启用但未设置API密钥，将在实际使用时抛出异常");
            return true;
        }
        
        return true;
    }
}